using System;
using System.IO;
using System.Globalization;
using Hexaly.Optimizer;

public class AdvertisingCampaign : IDisposable
{
    // Number of slots
    public int nbSlots;

    // Number of ads
    public int nbAds;

    // Ads profit
    public int[] adProfits;

    // Probability for an ad to reach its target audience during a slot 
    public double[][] probabilitiesData;

    // Hexaly Optimizer
    HexalyOptimizer optimizer;

    // Decision variables : for each ad, a set containing the slots assigned
    public HxExpression[] assignments;

    // Objective : maximize the total expected profit
    HxExpression totalProfit;

    public AdvertisingCampaign()
    {
        optimizer = new HexalyOptimizer();
    }

    public void Dispose()
    {
        if (optimizer != null)
            optimizer.Dispose();
    }

    public void ReadInstance(string fileName)
    {
        using (StreamReader input = new StreamReader(fileName))
        {
            string[] firstLineSplitted = input.ReadLine().Split(' ');
            nbSlots = int.Parse(firstLineSplitted[0]);
            nbAds = int.Parse(firstLineSplitted[1]);

            adProfits = new int[nbAds];
            for (int a = 0; a < nbAds; ++a)
            {
                adProfits[a] = int.Parse(input.ReadLine());
            }

            probabilitiesData = new double[nbSlots][];
            for (int s = 0; s < nbSlots; ++s)
            {
                probabilitiesData[s] = new double[nbAds];
            }
            for (int i = 0; i < nbSlots * nbAds; ++i)
            {
                string[] lineSplitted = input.ReadLine().Split(' ');
                int s = int.Parse(lineSplitted[0]);
                int a = int.Parse(lineSplitted[1]);
                probabilitiesData[s][a] = double.Parse(
                    lineSplitted[2],
                    CultureInfo.InvariantCulture
                );
            }
        }
    }

    public void Solve(int limit)
    {
        // Declare the optimization model
        HxModel model = optimizer.GetModel();

        // Variable declaration : set of slots assigned to each ad
        assignments = new HxExpression[nbAds];
        for (int a = 0; a < nbAds; ++a)
        {
            assignments[a] = model.Set(nbSlots);
        }

        // Each shot can only be assigned once
        model.Constraint(model.Partition(assignments));

        // Maximize the total profit :
        // 1 - probabilities[s][a] is the probability that ad a does not reach its audience during slot s
        // prod(assignments[a], s=>1 - probabilities[s][a]) is the probability that ad a does not reach its audience over the entire campaign
        // 1 - prod(assignments[a], s=>1 - probabilities[s][a]) is the probability that ad a reaches its audience over the entire campaign
        // AdProfits[a]*(1 - prod(assignments[a], s=>1 - probabilities[s][a])) is the expected profit of ad a over the entire campaign
        // If no argument is given, the "prod" operator returns the integer 1
        
        HxExpression probabilities = model.Array(probabilitiesData);
        HxExpression[] profit = new HxExpression[nbAds];

        for (int a = 0; a < nbAds; ++a)
        {
            HxExpression profitLambda = model.LambdaFunction(s => 1 - model.At(model.At(probabilities, s), a));
            profit[a] = adProfits[a] * (1 - model.Prod(assignments[a], profitLambda));
        }
        totalProfit = model.Sum(profit);

        // Objective 
        model.Maximize(totalProfit);
        model.Close();

        // Parametrize the optimizer
        optimizer.GetParam().SetTimeLimit(limit);
        optimizer.Solve();
    }

    // Write the solution in a file with the following format:
    // - instance name, time limit and total expected profit
    // - which slots were allocated to each ad
    public void WriteSolution(string infile, string outfile)
    {
        using (StreamWriter output = new StreamWriter(outfile))
        {
            output.WriteLine("File name: " + infile + "; totalProfit = " + totalProfit.GetDoubleValue());
            for (int a = 0; a < nbAds; ++a)
            {
                HxCollection assignment = assignments[a].GetCollectionValue();
                if (assignment.Count() == 0)
                    continue;
                output.Write("Add " + a + " assigned to ");
                for (int s = 0; s < assignment.Count(); ++s)
                {
                    output.Write("slot " + assignment[s] + ", ");
                }
                output.WriteLine("");
            }
        }
    }

    public static void Main(string[] args)
    {
        if (args.Length < 1)
        {
            Console.WriteLine("Usage: AdvertisingCampaign inputFile [solFile] [timeLimit]");
            Environment.Exit(1);
        }
        string instanceFile = args[0];
        string outputFile = args.Length > 1 ? args[1] : null;
        string strTimeLimit = args.Length > 2 ? args[2] : "60";

        using (AdvertisingCampaign model = new AdvertisingCampaign())
        {
            model.ReadInstance(instanceFile);
            model.Solve(int.Parse(strTimeLimit));
            if (outputFile != null)
                model.WriteSolution(instanceFile, outputFile);
        }
    }
}