import java.io.*;
import java.util.*;

import com.hexaly.optimizer.*;

public class AdvertisingCampaign {
    
    // Number of slots
    public int nbSlots;

    // Number of ads
    public int nbAds;

    // Ads profit
    public int[] AdProfitsData;

    // Probability for an ad to reach its target audience during a slot 
    public double[][] probabilitiesData;

    // Hexaly Optimizer
    public HexalyOptimizer optimizer;

    // Decision variables : for each ad, a set containing the slots assigned
    public HxExpression[] assignments;

    // Objective : maximize the total expected profit
    public HxExpression totalProfit;

    public AdvertisingCampaign(HexalyOptimizer optimizer) {
        this.optimizer = optimizer;
    }

    public void readInstance(String fileName) throws IOException {
        try (Scanner input = new Scanner(new File(fileName))) {
            input.useLocale(Locale.ROOT);
            nbSlots = input.nextInt();
            nbAds = input.nextInt();
            AdProfitsData = new int[nbAds];
            for (int a = 0; a < nbAds; ++a) {
                AdProfitsData[a] = input.nextInt();
            }

            probabilitiesData = new double[nbSlots][nbAds];
            for (int i = 0; i < nbSlots*nbAds; ++i) {
                int s = input.nextInt();
                int a = input.nextInt();
                double prob = input.nextDouble();
                probabilitiesData[s][a] = prob;
            }
        }
    }

    public void solve(int timeLimit) {
        // Declare the optimization model
        HxModel model = optimizer.getModel();

        // Variable declaration : set of slots assigned to each ad
        assignments = new HxExpression[nbAds];
        HxExpression assignmentsArray = model.array();
        for (int a = 0; a < nbAds; ++a) {
            assignments[a] = model.setVar(nbSlots);
            assignmentsArray.addOperand(assignments[a]);
        }

        // Unique ad affectation by slot
        model.constraint(model.partition(assignmentsArray));

        // Maximize the total profit :
        // 1 - probabilities[s][a] is the probability that ad a does not reach its audience during slot s
        // prod(assignments[a], s=>1 - probabilities[s][a]) is the probability that ad a does not reach its audience over the entire campaign
        // 1 - prod(assignments[a], s=>1 - probabilities[s][a]) is the probability that ad a reaches its audience over the entire campaign
        // AdProfits[a]*(1 - prod(assignments[a], s=>1 - probabilities[s][a])) is the expected profit of ad a over the entire campaign
        // If no argument is given, the "prod" operator returns the integer 1

        HxExpression AdProfits = model.array(AdProfitsData);
        HxExpression[] profit = new HxExpression[nbAds];
        HxExpression probabilities = model.array(probabilitiesData);
        for (int a = 0; a < nbAds; ++a) {
            HxExpression actualAd = model.createConstant(a);
            HxExpression profitLambda = model.lambdaFunction(s-> model.sub(1, model.at(model.at(probabilities, s), actualAd)));
            profit[a] = model.prod(model.at(AdProfits, a), model.sub(1, model.prod(assignments[a], profitLambda)));
        }

        totalProfit = model.sum(profit);
        model.maximize(totalProfit);
        model.close();

        optimizer.getParam().setTimeLimit(timeLimit);
        optimizer.solve();
    }

    // Write the solution in a file with the following format:
    // - instance name, time limit and total expected profit
    // - which slots were allocated to each ad
    public void writeSolution(String infile, String outfile) throws IOException{        
        try (PrintWriter output = new PrintWriter(outfile)) {
            output.println("File name: " + infile + "; totalProfit = " + totalProfit.getDoubleValue());
            for (int a = 0; a < nbAds; ++a) {
                HxCollection assignment = assignments[a].getCollectionValue();
                if (assignment.count() == 0)
                    continue;
                output.print("Ad " + a + " assigned to ");
                for (int s = 0; s < assignment.count(); ++s) {
                    output.print("slot " + assignment.get(s) + ", ");
                }
                 output.println("");
            }
        } catch (FileNotFoundException e) {
            System.out.println("An error occurred.");
            e.printStackTrace(); 
        }
    }

    public static void main(String[] args) {
        if (args.length < 1) {
            System.err.println("Usage: java main inputFile [outputFile] [timeLimit]");
            System.exit(1);
        }

        try (HexalyOptimizer optimizer = new HexalyOptimizer()) {
            String instanceFile = args[0];
            String strOutfile = args.length > 1 ? args[1] : null;
            String strTimeLimit = args.length > 2 ? args[2] : "60";

            AdvertisingCampaign model = new AdvertisingCampaign(optimizer);
            model.readInstance(instanceFile);
            model.solve(Integer.parseInt(strTimeLimit));
            if (strOutfile != null)
                model.writeSolution(instanceFile, strOutfile);
        } catch (Exception ex) {
            System.err.println(ex);
            ex.printStackTrace();
            throw new RuntimeException(ex);
        }
    }
}
    
