#include "optimizer/hexalyoptimizer.h"
#include <algorithm>
#include <fstream>
#include <iostream>
#include <limits>
#include <numeric>
#include <vector>

using namespace hexaly;
using namespace std;

class AdvertisingCampaign{
    public:
        // Number of slots
        int nbSlots;

        // Number of ads
        int nbAds;

        // Ads profit
        vector<int> adProfits;

        // Probability for an ad to reach its target audience during a slot 
        vector<vector<double>> probabilitiesData;

        // Hexaly Optimizer
        HexalyOptimizer optimizer;

        // Decision variables : for each ad, a set containing the slots assigned
        vector<HxExpression> assignments;

        // Objective : maximize the total expected profit
        HxExpression totalProfit;

        void readInstance(const string& fileName) {
            ifstream infile;
            infile.exceptions(ifstream::failbit | ifstream::badbit);
            infile.open(fileName.c_str());
            infile >> nbSlots;
            infile >> nbAds;
            adProfits.resize(nbAds);
            for (int a = 0; a < nbAds; ++a) {
                int value;
                infile >> value;
                adProfits[a] = value;
            }
            probabilitiesData.resize(nbSlots);
            for (int s = 0; s < nbSlots; ++s) {
                probabilitiesData[s].resize(nbAds);
            }
            for (int i = 0; i < nbAds*nbSlots; ++i) {
                int s;
                int a;
                double prob;
                infile >> s;
                infile >> a;
                infile >> prob;
                probabilitiesData[s][a] = prob;
            }
            infile.close();
        }
        
        void solve(int timeLimit) {

            // Declare the optimization model
            HxModel model = optimizer.getModel();
          
            // Variable declaration : set of slots assigned to each ad
            assignments.resize(nbAds);
            for (unsigned int a = 0; a < nbAds; ++a) {
                assignments[a] = model.setVar(nbSlots);
            }

            // Unique ad affectation by slot
            model.constraint(model.partition(assignments.begin(), assignments.end()));
            
            HxExpression probabilities = model.array();
            for (int s = 0; s < nbSlots; ++s){
                probabilities.addOperand(model.array(probabilitiesData[s].begin(), probabilitiesData[s].end()));
            }

            // Maximize the total profit :
            // 1 - probabilities[s][a] is the probability that ad a does not reach its audience during slot s
            // prod(assignments[a], s=>1 - probabilities[s][a]) is the probability that ad a does not reach its audience over the entire campaign
            // 1 - prod(assignments[a], s=>1 - probabilities[s][a]) is the probability that ad a reaches its audience over the entire campaign
            // AdProfits[a]*(1 - prod(assignments[a], s=>1 - probabilities[s][a])) is the expected profit of ad a over the entire campaign
            // If no argument is given, the "prod" operator returns the integer 1
            
            vector<HxExpression> profit;
            profit.reserve(nbAds);

            for (int a = 0; a < nbAds; ++a) {
                HxExpression profitLambda = model.createLambdaFunction([&](HxExpression s){return 1 - model.at(probabilities, s, a);});
                profit.push_back(adProfits[a] * (1 - model.prod(assignments[a], profitLambda)));
            }
            totalProfit = model.sum(profit.begin(), profit.end());
            
            // Objective
            model.maximize(totalProfit);

            model.close();

            // Parametrize the optimizer
            optimizer.getParam().setTimeLimit(timeLimit);
            optimizer.solve();
        }

        // Write the solution in a file with the following format:
        // - instance name, time limit and total expected profit
        // - which slots were allocated to each ad
        void writeSolution(const string& fileName, const string& instanceFile, const string& timeLimit) {
            ofstream outfile;
            outfile.open(fileName.c_str());
            outfile << "Instance: " << instanceFile << " ; time_limit: " << timeLimit
                << " ; Objective value: " << totalProfit.getDoubleValue() << endl;
            for (int a = 0; a < nbAds; ++a) {
                HxCollection assignment = assignments[a].getCollectionValue();
                if (assignment.count() == 0)
                    continue;
                outfile << "Ad " << a << " assigned to " ;
                for (int s = 0; s < assignment.count(); ++s) {
                    outfile << "slot " << assignment[s] << ", ";
                }
                outfile << "" << endl;
            }
        }
};

int main(int argc, char** argv) {
    if (argc < 2) {
        cerr << "Usage: main inputFile [outputFile] [timeLimit]" << endl;
        return 1;
    }

    const char* instanceFile = argv[1];
    const char* outputFile = argc > 2 ? argv[2] : NULL;
    const char* strTimeLimit = argc > 3 ? argv[3] : "60";

    try {
        AdvertisingCampaign model;
        model.readInstance(instanceFile);
        model.solve(atoi(strTimeLimit));

        // If we want to write the solution
        if (outputFile != NULL)
            model.writeSolution(outputFile, instanceFile, strTimeLimit);
    } catch (const exception& e) {
        cerr << "An error occured: " << e.what() << endl;
    }
    return 0;
}