import hexaly.optimizer
import sys


def read_instances(instance_filename):
    with open(instance_filename) as f:
        lines = f.readlines()

    nb_slots = int(lines[0].split()[0])
    nb_ads = int(lines[0].split()[1])
    Ad_profits = [int(lines[i]) for i in range(1,nb_ads+1)]
    Probabilities_data = [[0.0 for i in range(nb_ads)] for j in range(nb_slots)]
    for line in lines[nb_ads+1:]:
        values = line.split()
        s = int(values[0])
        a = int(values[1])
        Probabilities_data[s][a] = float(values[2])
    return nb_slots, nb_ads, Ad_profits, Probabilities_data


def main(input_file, output_file, time_limit):

    # Read the instance
    nb_slots, nb_ads, Ad_profits, Probabilities_data = read_instances(input_file)
   
    with hexaly.optimizer.HexalyOptimizer() as optimizer:
        # Declare the model
        model = optimizer.model
        
        # Variable declaration : set of slots assigned to each ad
        assignments = [model.set(nb_slots) for a in range(nb_ads)]
        assignments_array = model.array(assignments)
        
        # Unique ad affectation by slot
        model.constraint(model.partition(assignments_array))

        # Maximize the total profit :
        # 1 - probabilities[s][a] is the probability that ad a does not reach its audience during slot s
        # prod(assignments[a], s=>1 - probabilities[s][a]) is the probability that ad a does not reach its audience over the entire campaign
        # 1 - prod(assignments[a], s=>1 - probabilities[s][a]) is the probability that ad a reaches its audience over the entire campaign
        # AdProfits[a]*(1 - prod(assignments[a], s=>1 - probabilities[s][a])) is the expected profit of ad a over the entire campaign
        # If no argument is given, the "prod" operator returns the integer 1
        
        profit = [None] * nb_ads
        Probabilities = model.array(Probabilities_data)
        
        for a in range(nb_ads):
            profit_lambda = model.lambda_function(lambda s : 1 - model.at(Probabilities, s, a))
            profit[a] = Ad_profits[a] * (1 - model.prod(assignments[a], profit_lambda))
        total_profit = model.sum(profit)

        # Objective
        model.maximize(total_profit)
        model.close()

        # Parameterize the optimizer
        optimizer.param.time_limit = time_limit
        optimizer.solve()

        # Write the solution in a file with the following format:
        #  - nb slots, nb ads and the total expected profit
        #  - which slots were allocated to each ad
        if output_file != None:
            with open(output_file, 'w') as f:
                f.write("Nb slots = " + str(nb_slots) + "; Nb ads = " + str(nb_ads) +  "; Total profit = " +
                        str(total_profit.value) + "\n")
                for a in range(nb_ads):
                    if assignments[a].value.count() == 0:
                        continue
                    f.write("Ad " + str(a) + " assigned to ")
                    for s in assignments[a].value:
                        f.write("slot " + str(s) + ", ")
                    f.write("\n")
                
      
      
if __name__ == '__main__':
    input_file = sys.argv[1]
    output_file = sys.argv[2] if len(sys.argv) >= 3 else None
    time_limit = int(sys.argv[3]) if len(sys.argv) >= 4 else 60
    main(input_file, output_file, time_limit)