using System;
using System.IO;
using System.Linq;
using Hexaly.Optimizer;
using System.Collections.Generic;

public class AircraftDeicing : IDisposable
{
    // Number of gates
    private int nbGates;

    // Number of aircrafts
    private int nbAircrafts;

    // Size of the aircrafts
    private int[] aircraftsSize;

    // Arrival date of the aircrafts
    private int[] arrivalDates;

    // Number of trucks
    private int nbTrucks;

    // Bounds of the number trucks authorized per aircraft
    private int[] minTrucks;
    private int[] maxTrucks;

    // Durations normalised by the truck number of the aircraft
    private int[,] durations;

    // Setup Matrix: the time needed between two aircrafts to clean the bay
    private int[,] setupMatrix;

    // Horizon: trivial upper bound for the end times of the tasks
    private int H;

    // Hexaly Optimizer
    private HexalyOptimizer optimizer;

    // Decision variables: to each aircraft, we associate an interval of de-icing
    private HxExpression[] aircraftsIntervals;

    // Decision variable: to each aircraft, we associate a number of trucks
    private HxExpression[] aircraftsNbTrucks;

    // Decision variables: to each gate, we associate the 
    // list of the aircrafts that will be de-iced there
    private HxExpression[] gates;

    // Objective = minimize the makespan: end of the de-icing of the last aircraft
    private HxExpression makespan;

    public AircraftDeicing()
    {
        optimizer = new HexalyOptimizer();
    }

    public void ReadInstance(string fileName)
    {
        using (StreamReader input = new StreamReader(fileName))
        {
            nbGates = int.Parse(input.ReadLine());

            nbAircrafts = int.Parse(input.ReadLine());

            string[] line_sp = input.ReadLine().Split(' ', StringSplitOptions.RemoveEmptyEntries);
            aircraftsSize = Array.ConvertAll(line_sp, int.Parse);

            line_sp = input.ReadLine().Split(' ', StringSplitOptions.RemoveEmptyEntries);
            arrivalDates = Array.ConvertAll(line_sp, int.Parse);

            nbTrucks = int.Parse(input.ReadLine());

            line_sp = input.ReadLine().Split(' ', StringSplitOptions.RemoveEmptyEntries);
            minTrucks = Array.ConvertAll(line_sp, int.Parse);

            line_sp = input.ReadLine().Split(' ', StringSplitOptions.RemoveEmptyEntries);
            maxTrucks = Array.ConvertAll(line_sp, int.Parse);

            int maxGlobalTrucks = maxTrucks.Max();

            durations = new int[maxGlobalTrucks, nbAircrafts];
            for (int i=0; i<maxGlobalTrucks; ++i)
            {
                line_sp = input.ReadLine().Split(' ', StringSplitOptions.RemoveEmptyEntries);
                int[] lineDuration = Array.ConvertAll(line_sp, int.Parse);
                for (int j = 0; j < nbAircrafts; ++j) durations[i, j] = lineDuration[j];
            }

            int baseLineDurationSum = 0;
            for (int i = 0; i < nbAircrafts; ++i) baseLineDurationSum += durations[0, i];
            H = arrivalDates.Max() + baseLineDurationSum;



            List<int[]> setupMatrixTemp = new List<int[]>();
            string line;
            while ((line = input.ReadLine()) != null)
            {
                string[] valuesStr = line.Split(new char[] { ' ', '\t' },
                        StringSplitOptions.RemoveEmptyEntries);

                int[] values = Array.ConvertAll(valuesStr, int.Parse);

                setupMatrixTemp.Add(values);
            }

            int nbLines = setupMatrixTemp.Count;
            setupMatrix = new int[nbLines, nbLines];

            for (int i = 0; i < nbLines; ++i)
            {
                for (int j = 0; j < nbLines; ++j) setupMatrix[i, j] = setupMatrixTemp[i][j];
            }
        }
    }

    public void Dispose()
    {
        optimizer.Dispose();
    }

    public void Solve(int timeLimit)
    {
        // Declare the optimization model
        HxModel model = optimizer.GetModel();

        // Decision Variable: time range for each aircraft
        aircraftsIntervals = new HxExpression[nbAircrafts];
        for (int aircraft = 0; aircraft < nbAircrafts; ++aircraft)
        {
            aircraftsIntervals[aircraft] = model.Interval(arrivalDates[aircraft], H);
        }
        // Create a HexalyOptimizer array in order to be able to access it with "at" operators
        HxExpression aircraftsIntervalsArray = model.Array(aircraftsIntervals);

        // Decision Variable: number of trucks used for each aircraft
        aircraftsNbTrucks = new HxExpression[nbAircrafts];
        for (int aircraft = 0; aircraft < nbAircrafts; ++aircraft)
        {
            aircraftsNbTrucks[aircraft] = model.Int(minTrucks[aircraftsSize[aircraft]],
                    maxTrucks[aircraftsSize[aircraft]]);
        }
        // Create a HexalyOptimizer array in order to be able to access it with "at" operators
        HxExpression aircraftsNbTrucksArray = model.Array(aircraftsNbTrucks);

        // Decision Variable: number of trucks used for each aircraft
        gates = new HxExpression[nbGates];
        for (int i = 0; i < nbGates; ++i) gates[i] = model.List(nbAircrafts);

        // Create a HexalyOptimizer array in order to be able to access it with "at" operators
        HxExpression gatesArray = model.Array(gates);

        // Makespan: end of the de-icing of the last aircraft
        makespan = model.Max();
        for (int i = 0; i < nbAircrafts; ++i) makespan.AddOperand(model.End(aircraftsIntervals[i]));

        // Create a HexalyOptimizer array in order to be able to access it with "at" operators 
        HxExpression setupMatrixArray = model.Array(setupMatrix);

        // Create a HexalyOptimizer array in order to be able to access it with "at" operators
        HxExpression durationsArray = model.Array(durations);

        // Constraint: every aircraft has a unique gate
        model.Constraint(model.Partition(gatesArray));

        // Constraint: duration of the de-icing
        for (int aircraft = 0; aircraft < nbAircrafts; ++aircraft)
        {
            HxExpression constraintDuration = (model.Length(aircraftsIntervals[aircraft])
                    == durationsArray[aircraftsNbTrucks[aircraft]-1][aircraft]);
            model.Constraint(constraintDuration);
        }

        // Constraint no-overlapping: one aircraft per gate for each time, and a setup time
        // between two aircrafts on the same gate
        for (int j = 0; j < nbGates; ++j)
        {
            HxExpression gate = gates[j];
            HxExpression constraintOverlapping = model.LambdaFunction(i =>
                    model.End(aircraftsIntervalsArray[gate[i]])
                            + setupMatrixArray[gate[i]][gate[i+1]]
                    <= model.Start(aircraftsIntervalsArray[gate[i+1]])
            );
            model.Constraint(model.And(model.Range(0, model.Count(gate)-1), constraintOverlapping));
        }

        // Constraint: trucks capacity
        HxExpression respectCapacity = model.LambdaFunction(time =>
        {
            HxExpression currNbTrucks = model.Sum();
            for (int i = 0; i< nbAircrafts; i++)
            {
                currNbTrucks.AddOperand(model.Contains(aircraftsIntervalsArray[i], time)
                    * aircraftsNbTrucksArray[i]
                    );
            }
            return currNbTrucks <= nbTrucks;
        });
        model.Constraint(model.And(model.Range(0, makespan), respectCapacity));

        // Minimize the makespan
        model.Minimize(makespan);

        model.Close();

        // Parameterize the optimizer
        optimizer.GetParam().SetTimeLimit(timeLimit);

        optimizer.Solve();
    }

/* Write the solution in a file with the following format:
     *  - the total makespan
     *  - for each aircraft, the bay, the starting date of the interval, 
            the ending date of the interval, the number of trucks attributed */
    public void WriteSolution(string fileName)
    {
        using (StreamWriter output = new StreamWriter(fileName))
        {
            Console.WriteLine("Solution written in file " + fileName);

            output.Write(makespan.GetValue);
            output.WriteLine();
            for (int gate = 0; gate < nbGates; ++gate)
            {
                HxCollection aircraftsCollection = gates[gate].GetCollectionValue();
                for (int i = 0; i < aircraftsCollection.Count(); ++i)
                {
                    long aircraft = aircraftsCollection[i];
                    output.Write(gate + "\t"
                            + aircraftsIntervals[aircraft].GetIntervalValue().Start() + "\t"
                            + aircraftsIntervals[aircraft].GetIntervalValue().End() + "\t"
                            + aircraftsNbTrucks[aircraft].GetIntValue());
                    output.WriteLine();
                }
            }
        }
    }

    public static void Main(string[] args)
    {
        if (args.Length < 1)
        {
            Console.WriteLine("Usage: AircraftDeicing instanceFile [outputFile] [timeLimit]");
            System.Environment.Exit(1);
        }

        string instanceFile = args[0];
        string outputFile = args.Length > 1 ? args[1] : null;
        string strTimeLimit = args.Length > 2 ? args[2] : "60";

        using (AircraftDeicing model = new AircraftDeicing())
        {
            model.ReadInstance(instanceFile);
            model.Solve(int.Parse(strTimeLimit));
            if (outputFile != null) model.WriteSolution(outputFile);
        }
    }
}