import java.util.*;
import java.io.*;
import com.hexaly.optimizer.*;

public class AircraftDeicing {
    // Number of gates
    private int nbGates;

    // Number of aircrafts
    private int nbAircrafts;

    // Size of the aircrafts
    private int[] aircraftsSize;

    // Arrival date of the aircrafts
    private int[] arrivalDates;

    // Number of trucks
    private int nbTrucks;

    // Number of trucks possible per aircraft
    private int[] minTrucks;
    private int[] maxTrucks;

    // Durations normaised by the truck number of the aircraft
    private int[][] durations;

    // Setup Matirx: the time needed between two aircrafts to clean the bay
    private int[][] setupMatrix;

    // Horizon: trivial upper bound for the end times of the tasks
    private int H;

    // Hexaly Optimizer
    final HexalyOptimizer optimizer;
    // Decision variables: to each aircraft, we associate an interval of de-icing
    private HxExpression[] aircraftsIntervals;
    // Decision variable: to each aircraft, we associate a number of trucks
    private HxExpression[] aircraftsNbTrucks;
    // Decision variables: to each gate, we associate the list of aircrafts to be de-iced there
    private HxExpression[] gates;
    // Objective = minimize the makespan: end of the de-icing of the last aircraft
    private HxExpression makespan;

    public AircraftDeicing(HexalyOptimizer optimizer) throws IOException {
        this.optimizer = optimizer;
    }

    public static int maxInt(int[] v) {
        return Arrays.stream(v).max().getAsInt();
    }

    public void readInstance(String fileName) throws IOException {
        try (Scanner input = new Scanner(new File(fileName))) {
            input.useLocale(Locale.ROOT);

            nbGates = input.nextInt();
            nbAircrafts = input.nextInt();

            aircraftsSize = new int[nbAircrafts];
            for (int aircraft = 0; aircraft < nbAircrafts; ++aircraft) {
                aircraftsSize[aircraft] = input.nextInt();
            }

            arrivalDates = new int[nbAircrafts];
            for (int aircraft = 0; aircraft < nbAircrafts; ++aircraft) {
                arrivalDates[aircraft] = input.nextInt();
            }

            nbTrucks = input.nextInt();

            minTrucks = new int[nbAircrafts];
            for (int i = 0; i < nbAircrafts; ++i) minTrucks[i] = input.nextInt();

            maxTrucks = new int[nbAircrafts];
            for (int i = 0; i < nbAircrafts; ++i) maxTrucks[i] = input.nextInt();

            int maxGlobalTrucks = maxInt(maxTrucks);

            durations = new int[maxGlobalTrucks][nbAircrafts];
            for (int m = 0; m < maxGlobalTrucks; ++m) {
                for (int aircraft = 0; aircraft < nbAircrafts; ++aircraft) {
                    durations[m][aircraft] = input.nextInt();
                }
            }

            H = maxInt(arrivalDates) + Arrays.stream(durations[0]).sum();

            setupMatrix = new int[nbAircrafts][nbAircrafts];
            for (int i = 0; i < nbAircrafts; ++i) {
                int[] col = new int[nbAircrafts];
                for (int j = 0; j < nbAircrafts; ++j) col[j] = (input.nextInt());
                setupMatrix[i] = col;
            }
        }
    }


    public void solve(int timeLimit) {
        // Declare the optimization model
        HxModel model = optimizer.getModel();

        // Decision Variable: time range for each aircraft
        aircraftsIntervals = new HxExpression[nbAircrafts];
        for (int aircraft = 0; aircraft < nbAircrafts; ++aircraft) {
            aircraftsIntervals[aircraft] = model.intervalVar(arrivalDates[aircraft], H);
        }
        // Create a HexalyOptimizer array in order to be able to access it with "at" operators
        HxExpression aircraftsIntervalsArray = model.array(aircraftsIntervals);

        // Decision Variable: number of trucks used for each aircraft
        aircraftsNbTrucks = new HxExpression[nbAircrafts];
        for (int aircraft = 0; aircraft < nbAircrafts; ++aircraft) {
            aircraftsNbTrucks[aircraft] = model.intVar(
                    minTrucks[aircraftsSize[aircraft]], maxTrucks[aircraftsSize[aircraft]]);
        }
        // Create a HexalyOptimizer array in order to be able to access it with "at" operators
        HxExpression aircraftsNbTrucksArray = model.array(aircraftsNbTrucks);

        // Decision Variable: number of trucks used for each aircraft
        gates = new HxExpression[nbGates];
        for (int i = 0; i < nbGates; ++i) gates[i] = model.listVar(nbAircrafts);

        // Create a HexalyOptimizer array in order to be able to access it with "at" operators
        HxExpression gatesArray = model.array(gates);

        // Create a HexalyOptimizer array in order to be able to access it with "at" operators 
        HxExpression setupMatrixArray = model.array(setupMatrix);

        // Create a HexalyOptimizer array in order to be able to access it with "at" operators
        HxExpression durationsArray = model.array(durations);

        // Makespan: end of the de-icing of the last aircraft
        makespan = model.max();
        for (int i = 0; i < nbAircrafts; ++i) makespan.addOperand(model.end(aircraftsIntervals[i]));

        // Constraint: every aircraft has a unique gate
        model.constraint(model.partition(gatesArray));

        // Constraint: duration of the de-icing
        for (int aircraft = 0; aircraft < nbAircrafts; ++aircraft) {
            HxExpression constraintDuration = model.eq(model.length(aircraftsIntervals[aircraft]),
                    model.at(model.at(durationsArray, model.sum(aircraftsNbTrucks[aircraft], -1)), aircraft));
            model.constraint(constraintDuration);
        }

        // Constraint no-overlapping: one aircraft per gate for each time, and a setup time
        // between two aircrafts on the same gate
        for (int j = 0; j < nbGates; ++j) {
            HxExpression gate = gates[j];
            HxExpression constraintOverlapping = model.lambdaFunction(i -> {
                HxExpression nextAvailability = model.sum();
                nextAvailability.addOperand(
                        model.end(model.at(aircraftsIntervalsArray, model.at(gate,i))));
                nextAvailability.addOperand(
                        model.at(setupMatrixArray, model.at(gate, i), 
                        model.at(gate, model.sum(i, 1))));
                return model.leq(nextAvailability, model.start(
                        model.at(aircraftsIntervalsArray, model.at(gate, model.sum(i, 1)))));
        });
            model.constraint(model.and(
                    model.range(0, model.sum(model.count(gate), -1)), constraintOverlapping));
        }

        // Constraint: trucks capacity
        HxExpression respectCapacity = model.lambdaFunction(time -> {
            HxExpression currNbTrucks = model.sum();
            for (int i = 0; i < nbAircrafts; i++) {
                currNbTrucks.addOperand(model.prod(model.contains(model.at(aircraftsIntervalsArray, i), time), 
                        model.at(aircraftsNbTrucksArray, i)));
            }
            return model.leq(currNbTrucks, nbTrucks);
        }
        );
        model.constraint(model.and(model.range(0, makespan), respectCapacity));

        // Minimization of the makespan
        model.minimize(makespan);

        model.close();

        // Parameterize the optimizer
        optimizer.getParam().setTimeLimit(timeLimit);

        optimizer.solve();
    }

/* Write the solution in a file with the following format:
     *  - the total makespan
     *  - for each aircraft, the bay, the starting date of the interval,
     *        the ending date of the intrval, the number of trucks attributed */
    public void writeSolution(String fileName) throws IOException {
        try (PrintWriter output = new PrintWriter(fileName)) {
            System.out.println("Solution written in file " + fileName);

            output.write((int)makespan.getValue());
            output.write("\n");
            for (int gate = 0; gate < nbGates; ++gate) {
                HxCollection aircraftsCollection = gates[gate].getCollectionValue();
                for (int i = 0; i < aircraftsCollection.count(); ++i) {
                    int aircraft = (int)aircraftsCollection.get(i);
                    output.write(gate + "\t" 
                            + aircraftsIntervals[aircraft].getIntervalValue().getStart() + "\t"
                            + aircraftsIntervals[aircraft].getIntervalValue().getEnd() + "\t"
                            + aircraftsNbTrucks[aircraft].getIntValue());
                    output.write("\n");
                }
            }
        }
    }

    public static void main(String[] args) {
        if (args.length < 1) {
            System.out.println("Usage: AircraftDeicing instanceFile [outputFile] [timeLimit]");
            System.exit(1);
        }
        String instanceFile = args[0];
        String outputFile = args.length > 1 ? args[1] : null;
        String strTimeLimit = args.length > 2 ? args[2] : "60";

        try (HexalyOptimizer optimizer = new HexalyOptimizer()) {
            AircraftDeicing model = new AircraftDeicing(optimizer);
            model.readInstance(instanceFile);
            model.solve(Integer.parseInt(strTimeLimit));
            if (outputFile != null) model.writeSolution(outputFile);
        } catch (Exception ex) {
            System.err.println(ex);
            ex.printStackTrace();
            System.exit(1);
        }
    }
}
