#include "optimizer/hexalyoptimizer.h"
#include <fstream>
#include <iostream>
#include <vector>
#include <numeric>
#include <algorithm>
#include <cmath>

using namespace hexaly;
using namespace std;

class AircraftDeicing {
private:
    // Number of gates
    int nbGates;

    // Number of aircrafts
    int nbAircrafts;

    // Size of the aircrafts
    vector<int> aircraftsSize;

    // Arrival date of the aircrafts
    vector<int> arrivalDates;

    // Number of trucks
    int nbTrucks;

    // Number of trucks possible per aircraft
    vector<int> minTrucks;
    vector<int> maxTrucks;
    int maxGlobalTrucks;

    // Durations normalised by the truck number of the aircraft
    vector<vector<int>> durationsData;

    // Setup matrix: the time needed between two aircrafts to clean the gate
    vector<vector<int>> setupMatrixData;

    // Horizon: trivial upper bound for the end times of the tasks
    int H;

    // Hexaly Optimizer
    HexalyOptimizer optimizer;

    // Decision variables: to each aircraft, we associate a de-icing interval and a number of trucks
    vector<HxExpression> aircraftsIntervals;
    vector<HxExpression> aircraftsNbTrucks;

    // To each gate, we associate the list of the aircrafts that will be de-iced there
    vector<HxExpression> gates;

    // Objective = minimize the makespan: end of the de-icing of the last aircraft
    HxExpression makespan; 

public:
        void readInstance(const string& inFileName);
        void solve(int timeLimit);
        void writeSolution(const string& fileName);
};


void AircraftDeicing::readInstance (const string& inFileName){
    ifstream infile;
    infile.exceptions(ifstream::failbit | ifstream::badbit);
    infile.open(inFileName.c_str());
    
    infile >> nbGates;
    infile >> nbAircrafts;

    int value;
    for (int i = 0; i < nbAircrafts; ++i) {
        infile >> value;
        aircraftsSize.push_back(value);
    }
    for (int i = 0; i < nbAircrafts; ++i) {
        infile >> value;
        arrivalDates.push_back(value);
    }

    infile >> nbTrucks;

    for (int i = 0; i < nbAircrafts; ++i) {
        infile >> value;
        minTrucks.push_back(value);
    }

    for (int i = 0; i < nbAircrafts; ++i) {
        infile >> value;
        maxTrucks.push_back(value);
    }

    maxGlobalTrucks = *max_element(maxTrucks.begin(), maxTrucks.end());

    for (unsigned int m=0; m<maxGlobalTrucks; m++) {
        durationsData.push_back(vector<int> {});
        for (unsigned int i = 0; i < nbAircrafts; i++) {
            infile >> value;
            durationsData[m].push_back(value);
        }
    }

    H = *max_element(arrivalDates.begin(), arrivalDates.end()) 
            + accumulate(durationsData[0].begin(), durationsData[0].end(), 0);

    for (int i = 0; i < nbAircrafts; ++i) {
        setupMatrixData.push_back(vector<int> {});
        for (int j = 0; j < nbAircrafts; ++j) {
            infile >> value;
            setupMatrixData[i].push_back(value);
        }
    }
    
    infile.close();
}


void AircraftDeicing::solve(int timeLimit) {
    // Declare the optimization model
    HxModel model = optimizer.getModel();

    // Decision Variable: time range for each aircraft
    for (int aircraft = 0; aircraft < nbAircrafts; ++aircraft) {
        aircraftsIntervals.push_back(model.intervalVar(arrivalDates[aircraft], H));
    }
    HxExpression aircraftsIntervalsArray 
            = model.array(aircraftsIntervals.begin(), aircraftsIntervals.end());

    // Decision Variable: number of trucks used for each aircraft
    for (int aircraft = 0; aircraft < nbAircrafts; aircraft++) {
        aircraftsNbTrucks.push_back(
                model.intVar(minTrucks[aircraftsSize[aircraft]], 
                             maxTrucks[aircraftsSize[aircraft]]));
    }
    HxExpression aircraftsNbTrucksArray 
            = model.array(aircraftsNbTrucks.begin(), aircraftsNbTrucks.end());

    // Decision Variable: attribution of a gate to each aircraft
    for (int i = 0; i < nbGates; ++i) gates.push_back(model.listVar(nbAircrafts));
    HxExpression gatesArray = model.array(gates.begin(), gates.end());

    // Makespan: end of the de-icing of the last aircraft
    makespan = model.max();
    for (int i = 0; i < nbAircrafts; ++i) makespan.addOperand(model.end(aircraftsIntervals[i]));

    // Conversion of durations from vector<vector<int>> to HxExpression (array) 
    // to be able to access it with "at" operators
    HxExpression durations = model.array();
    for (int m = 0; m < maxGlobalTrucks; ++m) {
        durations.addOperand(
            model.array(durationsData[m].begin(), durationsData[m].end())
        );
    }

    // Conversion of setupMatrix from vector<vector<int>> to HxExpression (array)
    // to be able to access it with "at" operators
    HxExpression setupMatrix = model.array();
    for (int i = 0; i < nbAircrafts; ++i) {
        setupMatrix.addOperand(
            model.array(setupMatrixData[i].begin(), setupMatrixData[i].end())
        );
    }

    // Constraint: every aircraft has a unique gate
    model.constraint(model.partition(gates.begin(), gates.end()));

    // Constraint: duration of the de-icing
    for (int aircraft = 0; aircraft < nbAircrafts; aircraft++) {
        HxExpression constraintDuration = 
                model.length(aircraftsIntervals[aircraft]) 
                        == durations[aircraftsNbTrucks[aircraft]-1][aircraft];
        model.constraint(constraintDuration);
    }

    // Constraint no-overlapping: one aircraft per gate for each time, and a setup time
    // between two aircrafts on the same gate
    for (int j = 0; j < nbGates; j++) {
        HxExpression gate = gates[j];
        HxExpression constraintOverlapping = model.lambdaFunction([&](HxExpression i) {
            return (model.end(aircraftsIntervalsArray[gate[i]])
                    + setupMatrix[gate[i]][gate[i+1]]
                    <= model.start(aircraftsIntervalsArray[gate[i+1]]));
        }
        );
        model.constraint(model.and_(model.range(0, model.count(gate)-1), constraintOverlapping));
    }

    // Constraint: trucks capacity
    
    HxExpression respectCapacity = model.lambdaFunction([&](HxExpression time) {
        HxExpression currNbTrucks = model.sum();;
        for (int i = 0; i < nbAircrafts; i++){
            currNbTrucks.addOperand(model.contains(aircraftsIntervalsArray[i], time) * aircraftsNbTrucksArray[i]);
        }
        return currNbTrucks <= nbTrucks;
    });
    model.constraint(model.and_(model.range(0, makespan), respectCapacity));

    // Minimize the Makespan
    model.minimize(makespan);

    
    model.close();
    
    // Parameterize the optimizer
    optimizer.getParam().setTimeLimit(timeLimit);

    optimizer.solve();
}


/* Write the solution in a file with the following format:
     *  - the total makespan
     *  - for each aircraft, the gate, the starting date of the interval,
     *      the ending date of the interval,the number of trucks attributed */
void AircraftDeicing::writeSolution(const string& fileName) {
    ofstream outfile;
    outfile.exceptions(ofstream::failbit | ofstream::badbit);
    outfile.open(fileName.c_str());

    outfile << makespan.getValue() << endl;

    for (int gate = 0; gate < nbGates; ++gate) {
        HxCollection aircraftsCollection = gates[gate].getCollectionValue();
        for (int i = 0; i < aircraftsCollection.count(); ++i) {
            int aircraft = aircraftsCollection[i];
            outfile << gate << "\t" 
                    << aircraftsIntervals[aircraft].getIntervalValue().getStart() << "\t" 
                    << aircraftsIntervals[aircraft].getIntervalValue().getEnd() <<"\t" 
                    << aircraftsNbTrucks[aircraft].getIntValue() << endl;
        }
    }
}



int main(int argc, char** argv) {
    if (argc < 2) {
        std::cout << "Usage: aircraft_deicing instanceFile [outputFile] [timeLimit]" << std::endl;
        exit(1);
    }
    
    const char* instanceFile = argv[1];
    string inFileName = string(instanceFile);
    const char* outputFile = argc > 2 ? argv[2] : NULL;
    const char* strTimeLimit = argc > 3 ? argv[3] : "60";

    try {
        AircraftDeicing model;
        model.readInstance(inFileName);
        int TimeLimit = atoi(strTimeLimit);
        model.solve(TimeLimit);
        if (outputFile != NULL) {
            cout << "output file : " << outputFile << endl;
            model.writeSolution(outputFile);
        }
        return 0;
    } catch (const std::exception& e) {
        std::cerr << "An error occurred: " << e.what() << std::endl;
        return 1;
    }
}

