import hexaly.optimizer
import sys
import math

def main(instance_file, output_file, time_limit):
    with hexaly.optimizer.HexalyOptimizer() as optimizer:
        #
        # Declare the optimization model
        #
        model = optimizer.model

        with open(instance_file) as f:
            lines = f.readlines()

        # Number of gates
        nb_gates = int(lines[0])

        # Aircrafts data
        nb_aircrafts = int(lines[1])
        size_aircraft = [int(i) for i in lines[2].split()]
        arrival_date = [int(i) for i in lines[3].split()]

        # Trucks data
        nb_trucks = int(lines[4])
        min_trucks = [int(i) for i in lines[5].split()]
        max_trucks = [int(i) for i in lines[6].split()]
        max_total_trucks = max(max_trucks)

        # Duration data
        # base_line_duration = [int(i) for i in lines[7].split()]
        # ratio = [math.sqrt(k) for k in range(1, max_total_trucks + 1)]
        # duration = model.array([[math.ceil(base_line_duration[i]/ratio[m]) \
        #                     for m in range(max_total_trucks)] for i in range(nb_aircrafts)])
        
        duration = []
        for line in lines[7:12]:
            line_split = [int(i) for i in line.split()]
            duration.append(line_split)

        # Horizon: maximum duration
        H = max(arrival_date) + sum(duration[0])

        duration = model.array(duration)

        # Setup Matrix: times needed between two aircrafts in a gate
        setup_matrix = []
        for line in lines[12:]:
            line_split = [int(i) for i in line.split()]
            setup_matrix.append(line_split)
        setup_matrix = model.array(setup_matrix)

        # Decision variables: to each aircraft, we associate an interval of de-icing 
        # and a number of trucks.
        # A aircraft cannot be de-iced before its arrival
        aircraft_intervals = model.array( \
                [model.interval(arrival_date[aircraft], H) for aircraft in range(nb_aircrafts)])
        aircraft_nb_trucks = model.array( \
                [model.int(min_trucks[size_aircraft[i]], max_trucks[size_aircraft[i]]) \
                for i in range(nb_aircrafts)])

        # Decision variables: to each gate, we associate the list of 
        # the aircrafts that will be de-iced there.
        gates = model.array([model.list(nb_aircrafts) for _ in range(nb_gates)])

        # Constraint: every aircraft has a unique gate
        model.constraint(model.partition(gates))

        # Constraint non-overlap: one aircraft per gate for each time, and a setup time
        # between two aircrafts on the same gate
        for i in range(nb_gates):
            gate = gates[i]
            constraint_overlapping = model.lambda_function(
                lambda i : model.end(aircraft_intervals[gate[i]]) + setup_matrix[gate[i]][gate[i+1]] \
                        <= model.start(aircraft_intervals[gate[i+1]]))
            model.constraint(model.and_( \
                    model.range(0, model.count(gate)-1), constraint_overlapping))

        # Constraint: duration of the de-icing
        for aircraft in range(nb_aircrafts):
            constraint_duration = model.length(aircraft_intervals[aircraft]) \
                                    == duration[aircraft_nb_trucks[aircraft]-1][aircraft]
            model.constraint(constraint_duration)

        # Makespan: time of the end of all the de-icing tasks
        makespan = model.max([model.end(aircraft_intervals[aircraft]) \
                                    for aircraft in range(nb_aircrafts)])

        # Constraint: no more than nb_trucks in total
        constraint_capacity = model.lambda_function(
            lambda time : model.sum(model.contains(aircraft_intervals[aircraft], time) \
                    * aircraft_nb_trucks[aircraft] for aircraft in range(0, nb_aircrafts)) \
                    <= nb_trucks
        )
        model.constraint(model.and_(model.range(0, makespan), constraint_capacity))
        
        model.minimize(makespan)

        model.close()

        # Parameterize the optimizer
        optimizer.param.time_limit = time_limit
        optimizer.solve()
        
        #
        # Write the solution in a file with the following format:
        #  -the total makespan
        #  - for each aircraft, the bay, the starting date of the interval, 
        #       the ending date of the interval, the number of trucks attributed
        if output_file != None:
            with open(output_file, "w") as f:
                print("Solution written in file ", output_file)
                f.write(str(makespan.value))
                f.write("\n")
                for b in range(nb_gates):
                        gate = gates.value[b]
                        for aircraft in gate:
                            f.write(str(b) + " " \
                                    + str(aircraft_intervals.value[aircraft].start()) + " " \
                                    + str(aircraft_intervals.value[aircraft].end()) + " " \
                                    + str(aircraft_nb_trucks.value[aircraft]))
                            f.write("\n")


if __name__ == '__main__':
    if len(sys.argv) < 2:
        print("Usage: python aircraft_deicing.py instance_file [output_file] [time_limit]")
        sys.exit(1)

    instance_file = sys.argv[1]
    output_file = sys.argv[2] if len(sys.argv) >= 3 else None
    time_limit = int(sys.argv[3]) if len(sys.argv) >= 4 else 60
    main(instance_file, output_file, time_limit)
