using System;
using System.IO;
using System.Linq;
using System.Collections.Generic;
using Hexaly.Optimizer;

public class BinPackingConflicts : IDisposable
{
    // Number of items
    int nbItems;

    // Capacity of each bin
    int binCapacity;

    // Maximum number of bins
    int nbMaxBins;

    // Minimum number of bins
    int nbMinBins;

    // Weight of each item
    long[] weightsData;

    // List of forbidden items
    private List<List<int>> forbiddenItems;

    // Hexaly Optimizer
    HexalyOptimizer optimizer;

    // Decision variables
    HxExpression[] bins;

    // Bin where the item is
    private HxExpression[] binForItem;

    // Weight of each bin in the solution
    HxExpression[] binWeights;

    // Whether the bin is used in the solution
    HxExpression[] binsUsed;

    // Objective
    HxExpression totalBinsUsed;

    public BinPackingConflicts()
    {
        optimizer = new HexalyOptimizer();
    }

    /* Read instance data */
    void ReadInstance(string fileName)
    {
        int count = 0;
            using (StreamReader input = new StreamReader(fileName))
            {
                forbiddenItems = new List<List<int>>();
                string firstLine = input.ReadLine();
                string[] firstLineParts = firstLine.Split(' ');
                nbItems = int.Parse(firstLineParts[0]);
                binCapacity = int.Parse(firstLineParts[1]);
                weightsData = new long[nbItems];

                string line;
                while ((line = input.ReadLine()) != null)
                {
                    string[] lineParts = line.Split(new[] { ' ' }, StringSplitOptions.RemoveEmptyEntries);
                    weightsData[count] = long.Parse(lineParts[1]);
                    forbiddenItems.Add(new List<int>());
                    for (int i = 2; i < lineParts.Length; ++i)
                    {
                        forbiddenItems[count].Add(int.Parse(lineParts[i]) - 1);
                    }
                    count++;
                }
            }
        nbMinBins = (int)Math.Ceiling((double)weightsData.Sum() / binCapacity);
        nbMaxBins = nbItems;
    }

    public void Dispose()
    {
        if (optimizer != null)
            optimizer.Dispose();
    }

    void Solve(int limit)
    {
        // Declare the optimization model
        HxModel model = optimizer.GetModel();

        bins = new HxExpression[nbMaxBins];
        binWeights = new HxExpression[nbMaxBins];
        binsUsed = new HxExpression[nbMaxBins];
        HxExpression binsArray = model.Array();
        HxExpression forbiddenItemsArray = model.Array();

        // Set decisions: bins[k] represents the items in bin k
        for (int k = 0; k < nbMaxBins; ++k)
        {
            bins[k] = model.Set(nbItems);
            binsArray.AddOperand(bins[k]);
        }

        for (int i = 0; i < nbItems; ++i)
        {
            forbiddenItemsArray.AddOperand(model.Array(forbiddenItems[i]));
        }

        // Find the bin where an item is packed
        binForItem = new HxExpression[nbItems];
        for (int i = 0; i < nbItems; ++i) {
            binForItem[i] = model.Find(binsArray, i);
        }

        // Each item must be in one bin and one bin only
        model.Constraint(model.Partition(bins));

        // Create an array and a function to retrieve the item's weight
        HxExpression weights = model.Array(weightsData);
        HxExpression weightLambda = model.LambdaFunction(i => weights[i]);

        for (int k = 0; k < nbMaxBins; ++k)
        {
            // Weight constraint for each bin
            binWeights[k] = model.Sum(bins[k], weightLambda);
            model.Constraint(binWeights[k] <= binCapacity);

            // Bin k is used if at least one item is in it
            binsUsed[k] = model.Count(bins[k]) > 0;
        }

        // Forbidden constraint for each items
        for (int i = 0; i < nbItems; ++i)
        {
            HxExpression itemsIntersection = model.Intersection(binsArray[binForItem[i]], forbiddenItemsArray[i]);
            model.Constraint(model.Count(itemsIntersection) == 0);
        }

        // Count the used bins
        totalBinsUsed = model.Sum(binsUsed);

        // Minimize the number of used bins
        model.Minimize(totalBinsUsed);

        model.Close();

        // Parametrize the optimizer
        optimizer.GetParam().SetTimeLimit(limit);

        // Stop the search if the lower threshold is reached
        optimizer.GetParam().SetObjectiveThreshold(0, nbMinBins);

        optimizer.Solve();
    }

    /* Write the solution in a file */
    void WriteSolution(string fileName)
    {
        using (StreamWriter output = new StreamWriter(fileName))
        {
            for (int k = 0; k < nbMaxBins; ++k)
            {
                if (binsUsed[k].GetValue() != 0)
                {
                    output.Write("Bin weight: " + binWeights[k].GetValue() + " | Items: ");
                    HxCollection binCollection = bins[k].GetCollectionValue();
                    for (int i = 0; i < binCollection.Count(); ++i)
                        output.Write(binCollection[i] + " ");
                    output.WriteLine();
                }
            }
        }
    }

    public static void Main(string[] args)
    {
        if (args.Length < 1)
        {
            Console.WriteLine("Usage: BinPackingConflicts inputFile [solFile] [timeLimit]");
            Environment.Exit(1);
        }
        string instanceFile = args[0];
        string outputFile = args.Length > 1 ? args[1] : null;
        string strTimeLimit = args.Length > 2 ? args[2] : "5";

        using (BinPackingConflicts model = new BinPackingConflicts())
        {
            model.ReadInstance(instanceFile);
            model.Solve(int.Parse(strTimeLimit));
            if (outputFile != null)
                model.WriteSolution(outputFile);
        }
    }
}
