import hexaly.optimizer
import sys
import math

if len(sys.argv) < 2:
    print("Usage: python bin_packing_conflicts.py inputFile [outputFile] [timeLimit]")
    sys.exit(1)


def read_integers(filename):
    with open(filename) as f:
        return [int(elem) for elem in f.read().split()]


with hexaly.optimizer.HexalyOptimizer() as optimizer:
    # Read instance data
    filename = sys.argv[1]
    count = 0
    weights_data = []
    forbidden_items = []
    with open(filename) as f:
        for line in f:
            line = line.split()
            if count == 0:
                nb_items = int(line[0])
                bin_capacity = int(line[1])
            else:
                weights_data.append(int(line[1]))
                forbidden_items.append([])
                for i in range(2, len(line)):
                    forbidden_items[count-1].append(int(line[i]) - 1)
            count += 1
                    
    nb_min_bins = int(math.ceil(sum(weights_data) / float(bin_capacity)))
    nb_max_bins = nb_items

    #
    # Declare the optimization model
    #
    model = optimizer.model

    # Set decisions: bins[k] represents the items in bin k
    bins = [model.set(nb_items) for _ in range(nb_max_bins)]

    # Transform bins and itemFordbidden list into hx expression
    bins_array = model.array(bins)
    forbidden_items_array = model.array(forbidden_items)

    # Find the bin where an item is packed
    bin_for_item = [model.find(bins_array, i) for i in range(nb_items)]

    # Each item must be in one bin and one bin only
    model.constraint(model.partition(bins))

    # Create an array and a function to retrieve the item's weight
    weights = model.array(weights_data)
    weight_lambda = model.lambda_function(lambda i: weights[i])

    # Forbidden constraint for each items
    for i in range(nb_items):
        items_intersection = model.intersection(forbidden_items_array[i], bins_array[bin_for_item[i]])
        model.constraint(model.count(items_intersection) == 0)

    # Weight constraint for each bin
    bin_weights = [model.sum(b, weight_lambda) for b in bins]
    for w in bin_weights:
        model.constraint(w <= bin_capacity)

    # Bin k is used if at least one item is in it
    bins_used = [model.count(b) > 0 for b in bins]

    # Count the used bins
    total_bins_used = model.sum(bins_used)

    # Minimize the number of used bins
    model.minimize(total_bins_used)
    model.close()

    # Parameterize the optimizer
    if len(sys.argv) >= 4:
        optimizer.param.time_limit = int(sys.argv[3])
    else:
        optimizer.param.time_limit = 5

    # Stop the search if the lower threshold is reached
    optimizer.param.set_objective_threshold(0, nb_min_bins)

    optimizer.solve()

    # Write the solution in a file
    if len(sys.argv) >= 3:
        with open(sys.argv[2], 'w') as f:
            for k in range(nb_items):
                f.write("item:%d Weight:%d" % (k, weights_data[k]))
                f.write("\n")
            for k in range(nb_max_bins):
                if bins_used[k].value:
                    f.write("Bin weight: %d | Items: " % bin_weights[k].value)
                    for e in bins[k].value:
                        f.write("%d " % e)
                    f.write("\n")
