using System;
using System.IO;
using Hexaly.Optimizer;

public class CantileveredBeam : IDisposable
{
    public class Evaluate {
        /* Constant declaration */
        const int P = 1000;
        const int E = 10000000;
        const int L = 36;
        public static double[] possibleValues = {0.1, 0.25, 0.35, 0.5, 0.65, 0.75, 0.9, 1.0};

        /* External function */
        public double[] Call(HxExternalArgumentValues argumentValues) {
            // Argument retrieval
            double fH = argumentValues.GetDoubleValue(0);
            double fh1 = possibleValues[(int)argumentValues.GetIntValue(1)];
            double fb1 = argumentValues.GetDoubleValue(2);
            double fb2 = argumentValues.GetDoubleValue(3);

            // Big computation
            double I = 1.0 / 12.0 * fb2 * Math.Pow(fH - 2 * fh1, 3) + 2 * (1.0 / 12.0
                * fb1 * Math.Pow(fh1, 3) + fb1 * fh1 * Math.Pow(fH - fh1, 2) / 4.0);

            // Constraint computations
            double g1 = P * L * fH / (2 * I);
            double g2 = P * Math.Pow(L, 3) / (3* E * I);

            // Objective function computation
            double f = (2 * fh1 * fb1 + (fH - 2 * fh1) * fb2) * L;

            return new double[]{g1, g2, f};
        }
    }

    // Hexaly Optimizer
    private HexalyOptimizer optimizer;

    // Hexaly Program variables
    private HxExpression H;
    private HxExpression h1;
    private HxExpression b1;
    private HxExpression b2;

    private HxExpression objective;

    public CantileveredBeam() {
        optimizer = new HexalyOptimizer();
    }

    public void Dispose() {
        if (optimizer != null)
            optimizer.Dispose();
    }

    public void Solve(int evaluationLimit) {
        // Declare the optimization model
        HxModel model = optimizer.GetModel();

        // Numerical decisions
        H = model.Float(3.0, 7.0);
        h1 = model.Int(0, 7);
        b1 = model.Float(2.0, 12.0);
        b2 = model.Float(0.1, 2.0);

        // Create and call the external function
        Evaluate evaluateFunction = new Evaluate();
        HxDoubleArrayExternalFunction func = new HxDoubleArrayExternalFunction(evaluateFunction.Call);
        HxExpression funcExpr = model.DoubleArrayExternalFunction(func);
        HxExpression results = model.Call(funcExpr, H, h1, b1, b2);

        // Enable surrogate modeling
        HxExternalContext context = funcExpr.GetExternalContext();
        HxSurrogateParameters surrogateParams = context.EnableSurrogateModeling();

        // Constraint on bending stress
        model.Constraint(results[0] <= 5000);
        // Constraint on deflection at the tip of the beam
        model.Constraint(results[1] <= 0.10);

        objective = results[2];
        model.Minimize(objective);
        model.Close();

        // Parameterize the optimizer
        surrogateParams.SetEvaluationLimit(evaluationLimit);
        optimizer.Solve();
    }

    /* Write the solution in a file with the following format:
    * - The value of the minimum found
    * - The location (H; h1; b1; b2) of the minimum
    */
    public void WriteSolution(string fileName) {
        using (StreamWriter output = new StreamWriter(fileName))
        {
            output.WriteLine("Objective value: " + objective.GetDoubleValue());
            output.WriteLine("Point (H;h1;b1;b2): (" + H.GetDoubleValue()
                + ";" + Evaluate.possibleValues[(int)h1.GetIntValue()] + ";" + b1.GetDoubleValue()
                + ";" + b2.GetDoubleValue() + ")");
        }
    }

    public static void Main(string[] args)
    {
        string outputFile = args.Length > 0 ? args[0] : null;
        string strEvaluationLimit = args.Length > 1 ? args[1] : "30";

        using (CantileveredBeam model = new CantileveredBeam())
        {
            model.Solve(int.Parse(strEvaluationLimit));
            if (outputFile != null)
                model.WriteSolution(outputFile);
        }
    }
}