import java.io.*;
import com.hexaly.optimizer.*;

public class CantileveredBeam {

    /* Constant declaration */
    final static int P = 1000;
    final static int E = 10000000;
    final static int L = 36;
    final static double[] possibleValues = {0.1, 0.25, 0.35, 0.5, 0.65, 0.75, 0.9, 1.0};

    /* External function */
    private static class Evaluate implements HxDoubleArrayExternalFunction {
        @Override
        public double[] call(HxExternalArgumentValues argumentValues){
            // Argument retrieval
            double fH = argumentValues.getDoubleValue(0);
            double fh1 = possibleValues[(int)argumentValues.getIntValue(1)];
            double fb1 = argumentValues.getDoubleValue(2);
            double fb2 = argumentValues.getDoubleValue(3);

            // Big computation
            double I = 1.0 / 12.0 * fb2 * Math.pow(fH - 2 * fh1, 3) + 2 * (1.0 / 12.0
                * fb1 * Math.pow(fh1, 3) + fb1 * fh1 * Math.pow(fH - fh1, 2) / 4.0);

            // Constraint computations
            double g1 = P * L * fH / (2 * I);
            double g2 = P * Math.pow(L, 3) / (3* E * I);

            // Objective function computation
            double f = (2 * fh1 * fb1 + (fH - 2 * fh1) * fb2) * L;

            return new double[]{g1, g2, f};
        }
    }

    // Hexaly Optimizer
    private final HexalyOptimizer optimizer;

    // Hexaly Program variables
    private HxExpression H;
    private HxExpression h1;
    private HxExpression b1;
    private HxExpression b2;

    private HxExpression objective;

    private CantileveredBeam(HexalyOptimizer optimizer) {
        this.optimizer = optimizer;
    }

    private void solve(int evaluationLimit) {
        // Declare the optimization model
        HxModel model = optimizer.getModel();

        // Numerical decisions
        H = model.floatVar(3.0, 7.0);
        h1 = model.intVar(0, 7);
        b1 = model.floatVar(2.0, 12.0);
        b2 = model.floatVar(0.1, 2.0);

        // Create and call the external function
        HxExpression func = model.doubleArrayExternalFunction(new Evaluate());
        HxExpression results = model.call(func, H, h1, b1, b2);

        // Enable surrogate modeling
        HxSurrogateParameters surrogateParams = func.getExternalContext().enableSurrogateModeling();

        // Constraint on bending stress
        model.constraint(model.leq(model.at(results, 0), 5000));
        // Constraint on deflection at the tip of the beam
        model.constraint(model.leq(model.at(results, 1), 0.10));

        objective = model.at(results, 2);
        model.minimize(objective);
        model.close();

        // Parameterize the optimizer
        surrogateParams.setEvaluationLimit(evaluationLimit);

        optimizer.solve();
    }

    /* Write the solution in a file with the following format:
    * - The value of the minimum found
    * - The location (H; h1; b1; b2) of the minimum
    */
    private void writeSolution(String fileName) throws IOException {
        try (PrintWriter output = new PrintWriter(fileName)) {
            output.println("Objective value: " + objective.getDoubleValue());
            output.println("Point (H;h1;b1;b2): (" + H.getDoubleValue()
                + ";" + possibleValues[(int)h1.getIntValue()] + ";" + b1.getDoubleValue()
                + ";" + b2.getDoubleValue() + ")");
        }
    }

    public static void main(String[] args) {
        String outputFile = args.length > 0 ? args[0] : null;
        String strEvaluationLimit = args.length > 1 ? args[1] : "30";

        try (HexalyOptimizer optimizer = new HexalyOptimizer()) {
            CantileveredBeam model = new CantileveredBeam(optimizer);
            model.solve(Integer.parseInt(strEvaluationLimit));
            if (outputFile != null) {
                model.writeSolution(outputFile);
            }
        } catch (Exception ex) {
            System.err.println(ex);
            ex.printStackTrace();
            System.exit(1);
        }
    }
}