#include "optimizer/hexalyoptimizer.h"
#include <vector>
#include <iostream>
#include <fstream>

using namespace std;
using namespace hexaly;

/* Constant declaration */
const int P = 1000;
const int E = 10000000;
const int L = 36;
const float possibleValues[8] = {0.1, 0.25, 0.35, 0.5, 0.65, 0.75, 0.9, 1.0};

/* External function */
class Evaluate : public HxArrayExternalFunction<hxdouble> {

    void call(const HxExternalArgumentValues& argumentValues,
                vector<hxdouble>& resultValues) override {
        // Argument retrieval
        hxdouble fH = argumentValues.getDoubleValue(0);
        hxdouble fh1 = possibleValues[argumentValues.getIntValue(1)];
        hxdouble fb1 = argumentValues.getDoubleValue(2);
        hxdouble fb2 = argumentValues.getDoubleValue(3);

        // Big computation
        hxdouble I = 1.0 / 12.0 * fb2 * pow(fH - 2 * fh1, 3) + 2 * (1.0 / 12.0
            * fb1 * pow(fh1, 3) + fb1 * fh1 * pow(fH - fh1, 2) / 4.0);

        // Constraint computations
        hxdouble g1 = P * L * fH / (2 * I);
        hxdouble g2 = P * pow(L, 3) / (3 * E * I);

        // Objective function computations
        hxdouble f = (2 * fh1 * fb1 + (fH - 2 * fh1) * fb2) * L;

        // Return results
        resultValues.clear();
        resultValues.push_back(g1);
        resultValues.push_back(g2);
        resultValues.push_back(f);
    }
};

class CantileveredBeam {
public:
    // Hexaly Optimizer
    HexalyOptimizer optimizer;

    // Hexaly Program variables
    HxExpression H;
    HxExpression h1;
    HxExpression b1;
    HxExpression b2;

    HxExpression objective;

    void solve(int evaluationLimit){
        // Declare the optimization model
        HxModel model = optimizer.getModel();

        // Numerical decisions
        H = model.floatVar(3.0, 7.0);
        h1 = model.intVar(0, 7);
        b1 = model.floatVar(2.0, 12.0);
        b2 = model.floatVar(0.1, 2.0);

        // Create and call the external function
        Evaluate evaluate;
        HxExpression func = model.createExternalFunction(&evaluate);
        HxExpression results = func(H, h1, b1, b2);

        // Enable surrogate modeling
        HxExternalContext context = func.getExternalContext();
        HxSurrogateParameters surrogateParams = context.enableSurrogateModeling();

        // Constraint on bending stress
        model.constraint(results[0] <= 5000);
        // Constraint on deflection at the tip of the beam
        model.constraint(results[1] <= 0.10);

        objective = results[2];
        model.minimize(objective);
        model.close();

        // Parameterize the optimizer
        surrogateParams.setEvaluationLimit(evaluationLimit);

        optimizer.solve();
    }

    /* Write the solution in a file with the following format:
    * - The value of the minimum found
    * - The location (H; h1; b1; b2) of the minimum
    */
    void writeSolution(const string& fileName) {
        ofstream outfile;
        outfile.exceptions(ofstream::failbit | ofstream::badbit);
        outfile.open(fileName.c_str());
        outfile << "Objective value: " << objective.getDoubleValue() << endl;
        outfile << "Point (H;h1;b1;b2): (" << H.getDoubleValue() << ";"
            << possibleValues[h1.getIntValue()] << ";" << b1.getDoubleValue()
            << ";" << b2.getDoubleValue() << ")";
    }
};

int main(int argc, char** argv){
    const char* solFile = argc > 1 ? argv[1] : NULL;
    const char* strEvaluationLimit = argc > 2 ? argv[2] : "30";

    try {
        CantileveredBeam model;
        model.solve(atoi(strEvaluationLimit));
        if (solFile != NULL)
            model.writeSolution(solFile);
    } catch (const exception& e) {
        cerr << "An error occurred: " << e.what() << endl;
        return 1;
    }
    return 0;
}
