using System;
using System.IO;
using System.Collections;
using System.Collections.Generic;
using System.Linq;
using Hexaly.Optimizer;

public class CapacitatedArcRouting : IDisposable
{
    // Data from the problem
    int nbTrucks;
    int truckCapacity;
    int nbNodes;
    int nbRequiredEdges;
    int nbRequiredNodes;
    int nbNotRequiredEdges;
    int depotNode;
    int[] demandsData;
    int[] costsData;
    List<int> requiredNodes;
    int[] originsData;
    int[] destinationsData;
    int[] distToDepotData;
    int[] distFromDepotData;
    int[][] nodesDistData;
    int[][] edgesDistData;
    int[][] nodeNeighborsData;

    // Hexaly Optimizer
    HexalyOptimizer optimizer;

    // Decision variables
    HxExpression[] edgesSequencesVars;
    HxExpression totalCost;

    public CapacitatedArcRouting()
    {
        optimizer = new HexalyOptimizer();
    }

    /* Read instance data */
    void ReadInstance(string fileName)
    {
        ReadInputCarp(fileName);
    }

    public void Dispose()
    {
        if (optimizer != null)
            optimizer.Dispose();
    }

    void Solve(int limit)
    {
        // Declare the optimization model
        HxModel model = optimizer.GetModel();

        // Sequence of edges visited and "serviced" by each truck
        edgesSequencesVars = new HxExpression[nbTrucks];
        for (int i = 0; i < nbTrucks; ++i)
        {
            // A sequence can contain an edge in one direction or its reverse
            edgesSequencesVars[i] = model.List(2 * nbRequiredEdges);
        }
        HxExpression edgesSequences = model.Array(edgesSequencesVars);

        // Creates distance and cost arrays to be able to access it with an "at" operator
        HxExpression demands = model.Array(demandsData);
        HxExpression costs = model.Array(costsData);
        HxExpression distFromDepot = model.Array(distFromDepotData);
        HxExpression distToDepot = model.Array(distToDepotData);
        HxExpression edgesDist = model.Array(edgesDistData);

        // An edge must be serviced by at most one truck
        model.Constraint(model.Disjoint(edgesSequences));

        // An edge can be travelled in both directions but its demand must be satisfied only once
        for (int i = 0; i < nbRequiredEdges; ++i)
        {
            model.Constraint(
                model.Contains(edgesSequences, 2 * i) + model.Contains(edgesSequences, 2 * i + 1)
                    == 1
            );
        }

        HxExpression[] routeDistances = new HxExpression[nbTrucks];
        for (int k = 0; k < nbTrucks; ++k)
        {
            HxExpression sequence = edgesSequencesVars[k];
            HxExpression c = model.Count(sequence);

            // Quantity in each truck
            HxExpression demandLambda = model.LambdaFunction(j => demands[j]);
            HxExpression routeQuantity = model.Sum(sequence, demandLambda);
            // Capacity constraint : a truck must not exceed its capacity
            model.Constraint(routeQuantity <= truckCapacity);

            // Distance travelled by each truck
            HxExpression distLambda = model.LambdaFunction(
                i => costs[sequence[i]] + edgesDist[sequence[i - 1], sequence[i]]
            );
            routeDistances[k] =
                model.Sum(model.Range(1, c), distLambda)
                + model.If(
                    c > 0,
                    costs[sequence[0]] + distFromDepot[sequence[0]] + distToDepot[sequence[c - 1]],
                    0
                );
        }

        // Total Distance travelled
        totalCost = model.Sum(routeDistances);

        // Objective : minimize the distance travelled
        model.Minimize(totalCost);

        model.Close();

        // Parametrize the optimizer
        optimizer.GetParam().SetTimeLimit(limit);

        optimizer.Solve();
    }

    /* Write the solution in a file with the following format:
    - total distance
    - number of routes
    - for each truck, the edges visited */
    void WriteSolution(string fileName)
    {
        using (StreamWriter output = new StreamWriter(fileName))
        {
            output.WriteLine("Objective function value : " + totalCost.GetValue());
            output.WriteLine("Number of routes : " + nbTrucks);
            for (int k = 0; k < nbTrucks; ++k)
            {
                output.Write("Sequence of truck " + (k + 1) + ": ");
                HxCollection sequence = edgesSequencesVars[k].GetCollectionValue();
                int c = sequence.Count();
                for (int i = 0; i < c; ++i)
                {
                    output.Write(
                        "("
                            + originsData[sequence[i]]
                            + ", "
                            + destinationsData[sequence[i]]
                            + ")  "
                    );
                }
                output.WriteLine();
            }
        }
    }

    public static void Main(string[] args)
    {
        if (args.Length < 1)
        {
            Console.WriteLine("Usage: CapacitatedArcRouting inputFile [solFile] [timeLimit]");
            Environment.Exit(1);
        }
        string instanceFile = args[0];
        string outputFile = args.Length > 1 ? args[1] : null;
        string strTimeLimit = args.Length > 2 ? args[2] : "20";

        using (CapacitatedArcRouting model = new CapacitatedArcRouting())
        {
            model.ReadInstance(instanceFile);
            model.Solve(int.Parse(strTimeLimit));
            if (outputFile != null)
                model.WriteSolution(outputFile);
        }
    }

    // The input files follow the format of the DIMACS challenge
    private void ReadInputCarp(string fileName)
    {
        using (StreamReader input = new StreamReader(fileName))
        {
            string[] splitted;
            for (int i = 0; i < 2; ++i)
                input.ReadLine();
            splitted = input.ReadLine().Split(':');
            nbNodes = int.Parse(splitted[1]);
            splitted = input.ReadLine().Split(':');
            nbRequiredEdges = int.Parse(splitted[1]);
            splitted = input.ReadLine().Split(':');
            nbNotRequiredEdges = int.Parse(splitted[1]);
            splitted = input.ReadLine().Split(':');
            nbTrucks = int.Parse(splitted[1]);
            splitted = input.ReadLine().Split(':');
            truckCapacity = int.Parse(splitted[1]);
            for (int i = 0; i < 2; ++i)
                input.ReadLine();
            // Even indices store direct edges, and odd indices store reverse edges
            originsData = new int[2 * nbRequiredEdges];
            destinationsData = new int[2 * nbRequiredEdges];
            demandsData = new int[2 * nbRequiredEdges];
            costsData = new int[2 * nbRequiredEdges];
            nodeNeighborsData = new int[nbNodes][];
            for (int i = 0; i < nbNodes; ++i)
                nodeNeighborsData[i] = new int[nbNodes];
            for (int i = 0; i < nbNodes; ++i)
            {
                for (int j = 0; j < nbNodes; ++j)
                    nodeNeighborsData[i][j] = 0;
            }
            requiredNodes = new List<int>();
            input.ReadLine();
            for (int i = 0; i < nbRequiredEdges; ++i)
            {
                splitted = input.ReadLine().Split(' ');
                int firstNode = int.Parse(splitted[2].Remove(splitted[2].Length - 1));
                int secondNode = int.Parse(splitted[3].Remove(splitted[3].Length - 1));
                int cost = int.Parse(splitted[7]);
                int demand = int.Parse(splitted[11]);
                costsData[2 * i] = cost;
                demandsData[2 * i] = demand;
                costsData[2 * i + 1] = cost;
                demandsData[2 * i + 1] = demand;
                originsData[2 * i] = firstNode;
                destinationsData[2 * i] = secondNode;
                originsData[2 * i + 1] = secondNode;
                destinationsData[2 * i + 1] = firstNode;
                if (!requiredNodes.Contains(firstNode))
                    requiredNodes.Add(firstNode);
                if (!requiredNodes.Contains(secondNode))
                    requiredNodes.Add(secondNode);
                nodeNeighborsData[firstNode - 1][secondNode - 1] = cost;
                nodeNeighborsData[secondNode - 1][firstNode - 1] = cost;
            }
            if (nbNotRequiredEdges > 0)
            {
                input.ReadLine();
                for (int i = 0; i < nbNotRequiredEdges; ++i)
                {
                    splitted = input.ReadLine().Split(' ');
                    int firstNode = int.Parse(splitted[2].Remove(splitted[2].Length - 1));
                    int secondNode = int.Parse(splitted[3].Remove(splitted[3].Length - 1));
                    int cost = int.Parse(splitted[7]);
                    nodeNeighborsData[firstNode - 1][secondNode - 1] = cost;
                    nodeNeighborsData[secondNode - 1][firstNode - 1] = cost;
                }
            }
            splitted = input.ReadLine().Split(':');
            depotNode = int.Parse(splitted[1]);
        }
        nbRequiredNodes = requiredNodes.Count;
        int[] shortestPath = new int[nbNodes];
        FindRequiredPaths(ref shortestPath);
        FindDistanceBetweenEdges();
        FindDistanceToDepot();
        FindDistanceFromDepot(ref shortestPath);
    }

    // Finds the shortest path from one node "origin" to all the other nodes of the graph
    // thanks to the Dijkstra's algorithm
    public int MinDistance(int[] shortestPath, bool[] sptSet)
    {
        // Initializes min value
        int min = int.MaxValue,
            minIndex = -1;
        for (int i = 0; i < nbNodes; ++i)
            if (sptSet[i] == false && shortestPath[i] <= min)
            {
                min = shortestPath[i];
                minIndex = i;
            }
        return minIndex;
    }

    public void ShortestPathFinder(int origin, ref int[] shortestPath)
    {
        int[] currentNeighbors;
        int currentNode,
            distance;
        bool[] sptSet = new bool[nbNodes];
        for (int i = 0; i < nbNodes; ++i)
        {
            shortestPath[i] = int.MaxValue;
            sptSet[i] = false;
        }
        shortestPath[origin - 1] = 0;
        for (int count = 0; count < nbNodes; ++count)
        {
            currentNode = MinDistance(shortestPath, sptSet);
            sptSet[currentNode] = true;
            currentNeighbors = nodeNeighborsData[currentNode];
            for (int neighbor = 0; neighbor < nbNodes; ++neighbor)
            {
                if (currentNeighbors[neighbor] != 0)
                {
                    distance = currentNeighbors[neighbor];
                    if (
                        !sptSet[neighbor]
                        && shortestPath[currentNode] + distance < shortestPath[neighbor]
                    )
                    {
                        shortestPath[neighbor] = shortestPath[currentNode] + distance;
                    }
                }
            }
        }
    }

    public void FindRequiredPaths(ref int[] shortestPath)
    {
        nodesDistData = new int[nbRequiredNodes][];
        for (int i = 0; i < nbRequiredNodes; ++i)
            nodesDistData[i] = new int[nbNodes];
        for (int i = 0; i < nbRequiredNodes; ++i)
        {
            int node = requiredNodes[i];
            ShortestPathFinder(node, ref shortestPath);
            for (int j = 0; j < nbNodes; ++j)
                nodesDistData[i][j] = shortestPath[j];
        }
    }

    public void FindDistanceBetweenEdges()
    {
        edgesDistData = new int[2 * nbRequiredEdges][];
        for (int i = 0; i < 2 * nbRequiredEdges; ++i)
            edgesDistData[i] = new int[2 * nbRequiredEdges];
        for (int i = 0; i < 2 * nbRequiredEdges; ++i)
        {
            for (int j = 0; j < 2 * nbRequiredEdges; ++j)
            {
                if (destinationsData[i] == originsData[j])
                    edgesDistData[i][j] = 0;
                else
                {
                    for (int k = 0; k < nbRequiredNodes; ++k)
                    {
                        if (requiredNodes[k] == destinationsData[i])
                            edgesDistData[i][j] = nodesDistData[k][originsData[j] - 1];
                    }
                }
            }
        }
    }

    public void FindDistanceToDepot()
    {
        distToDepotData = new int[2 * nbRequiredEdges];
        for (int i = 0; i < 2 * nbRequiredEdges; ++i)
        {
            if (destinationsData[i] == depotNode)
                distToDepotData[i] = 0;
            else
            {
                for (int k = 0; k < nbRequiredNodes; ++k)
                {
                    if (requiredNodes[k] == destinationsData[i])
                        distToDepotData[i] = nodesDistData[k][depotNode - 1];
                }
            }
        }
    }

    public void FindDistanceFromDepot(ref int[] shortestPath)
    {
        distFromDepotData = new int[2 * nbRequiredEdges];
        for (int i = 0; i < 2 * nbRequiredEdges; ++i)
        {
            if (depotNode == originsData[i])
                distFromDepotData[i] = 0;
            else
            {
                bool depotIsRequiredNode = false;
                for (int k = 0; k < nbRequiredNodes; ++k)
                {
                    if (requiredNodes[k] == depotNode)
                    {
                        depotIsRequiredNode = true;
                        distFromDepotData[i] = nodesDistData[k][originsData[i] - 1];
                    }
                }
                if (!depotIsRequiredNode)
                {
                    ShortestPathFinder(depotNode, ref shortestPath);
                    distFromDepotData[i] = shortestPath[originsData[i] - 1];
                }
            }
        }
    }
}
