import java.util.*;
import java.io.*;
import com.hexaly.optimizer.*;

public class CapacitatedArcRouting {

    // Data from the problem
    int nbTrucks;
    int truckCapacity;
    int nbNodes;
    int nbRequiredEdges;
    int nbRequiredNodes;
    int nbNotRequiredEdges;
    int depotNode;
    int[] demandsData;
    int[] costsData;
    int[] originsData;
    int[] destinationsData;
    List<Integer> requiredNodes;
    int[] distToDepotData;
    int[] distFromDepotData;
    int[][] nodesDistData;
    int[][] edgesDistData;
    int[][] nodeNeighborsData;

    // Hexaly Optimizer
    private final HexalyOptimizer optimizer;

    // Decision variables
    HxExpression[] edgesSequencesVars;
    HxExpression totalDistance;

    private CapacitatedArcRouting(HexalyOptimizer optimizer) {
        this.optimizer = optimizer;
    }

    private void solve(int limit) {
        // Declare the optimization model
        HxModel model = optimizer.getModel();

        // Sequence of edges visited and "serviced" by each truck
        edgesSequencesVars = new HxExpression[nbTrucks];
        for (int i = 0; i < nbTrucks; ++i) {
            edgesSequencesVars[i] = model.listVar(2 * nbRequiredEdges);
        }
        HxExpression edgesSequences = model.array(edgesSequencesVars);

        // Create distance and cost arrays to be able to access it with an "at" operator
        HxExpression demands = model.array(demandsData);
        HxExpression costs = model.array(costsData);
        HxExpression distFromDepot = model.array(distFromDepotData);
        HxExpression distToDepot = model.array(distToDepotData);
        HxExpression edgesDist = model.array(edgesDistData);

        // An edge must be serviced by at most one truck
        model.constraint(model.disjoint(edgesSequences));

        // An edge can be travelled in both directions but its demand must be satisfied
        // only once
        for (int i = 0; i < nbRequiredEdges; ++i) {
            model.constraint(model
                .eq(model.sum(model.contains(edgesSequences, 2 * i), model.contains(edgesSequences, (2 * i) + 1)), 1));
        }
        HxExpression[] routeDistances = new HxExpression[nbTrucks];
        for (int k = 0; k < nbTrucks; ++k) {
            HxExpression sequence = edgesSequencesVars[k];
            HxExpression c = model.count(sequence);

            // Quantity in each truck
            HxExpression demandLambda = model.lambdaFunction(j -> model.at(demands, j));
            HxExpression routeQuantity = model.sum(sequence, demandLambda);
            // Capacity constraint : a truck must not exceed its capacity
            model.constraint(model.leq(routeQuantity, truckCapacity));

            // Distance travelled by each truck
            HxExpression distLambda = model.lambdaFunction(i -> model.sum(model.at(costs, model.at(sequence, i)),
                model.at(edgesDist, model.at(sequence, model.sub(i, 1)), model.at(sequence, i))));
            routeDistances[k] = model.sum(model.sum(model.range(1, c), distLambda),
                model.iif(model.gt(c, 0),
                    model.sum(model.at(costs, model.at(sequence, 0)), model.at(distFromDepot, model.at(sequence, 0)),
                        model.at(distToDepot, model.at(sequence, model.sub(c, 1)))),
                    0));
        }

        // Total Distance travelled
        totalDistance = model.sum(routeDistances);

        // Objective : minimize the distance travelled
        model.minimize(totalDistance);

        model.close();

        // Parametrize the optimizer
        optimizer.getParam().setTimeLimit(limit);

        optimizer.solve();
    }

    // Write the solution in a file with the following format:
    // - total distance
    // - number of routes
    // - for each truck, the edges visited
    private void WriteSolution(String filename) throws IOException {
        try (PrintWriter output = new PrintWriter(filename)) {
            output.println("Objective function value : " + totalDistance.getValue());
            output.println("Number of routes : " + nbTrucks);
            for (int k = 0; k < nbTrucks; ++k) {
                output.print("Sequence of truck " + (k + 1) + ": ");
                HxCollection sequence = edgesSequencesVars[k].getCollectionValue();
                int c = sequence.count();
                for (int i = 0; i < c; ++i) {
                    output.print("(" + originsData[(int) sequence.get(i)] + ", "
                        + destinationsData[(int) sequence.get(i)] + ")  ");
                }
                output.println();
            }
        }
    }

    // The input files follow the format of the DIMACS challenge
    private void readInstance(String filename) throws IOException {
        try (Scanner input = new Scanner(new File(filename))) {
            input.useLocale(Locale.ROOT);
            String[] splitted;
            for (int i = 0; i < 2; ++i)
                input.nextLine();
            splitted = input.nextLine().split(":");
            nbNodes = Integer.parseInt(splitted[1].trim());
            splitted = input.nextLine().split(":");
            nbRequiredEdges = Integer.parseInt(splitted[1].trim());
            splitted = input.nextLine().split(":");
            nbNotRequiredEdges = Integer.parseInt(splitted[1].trim());
            splitted = input.nextLine().split(":");
            nbTrucks = Integer.parseInt(splitted[1].trim());
            splitted = input.nextLine().split(":");
            truckCapacity = Integer.parseInt(splitted[1].trim());
            for (int i = 0; i < 2; ++i)
                input.nextLine();
            requiredNodes = new ArrayList<Integer>();
            // Even indices store direct edges, and odd indices store reverse edges
            demandsData = new int[2 * nbRequiredEdges];
            costsData = new int[2 * nbRequiredEdges];
            originsData = new int[2 * nbRequiredEdges];
            destinationsData = new int[2 * nbRequiredEdges];
            nodeNeighborsData = new int[nbNodes][];
            for (int j = 0; j < nbNodes; ++j) {
                nodeNeighborsData[j] = new int[nbNodes];
            }
            input.nextLine();
            for (int i = 0; i < nbRequiredEdges; ++i) {
                splitted = input.nextLine().split(" ");
                int firstNode = Integer.parseInt(splitted[2].substring(0, splitted[2].length() - 1));
                int secondNode = Integer.parseInt(splitted[3].substring(0, splitted[3].length() - 1));
                int cost = Integer.parseInt(splitted[7]);
                int demand = Integer.parseInt(splitted[11]);
                costsData[2 * i] = cost;
                demandsData[2 * i] = demand;
                costsData[2 * i + 1] = cost;
                demandsData[2 * i + 1] = demand;
                originsData[2 * i] = firstNode;
                destinationsData[2 * i] = secondNode;
                originsData[2 * i + 1] = secondNode;
                destinationsData[2 * i + 1] = firstNode;
                if (!requiredNodes.contains(firstNode)) {
                    requiredNodes.add(firstNode);
                }
                if (!requiredNodes.contains(secondNode)) {
                    requiredNodes.add(secondNode);
                }
                nodeNeighborsData[firstNode - 1][secondNode - 1] = cost;
                nodeNeighborsData[secondNode - 1][firstNode - 1] = cost;
            }
            if (nbNotRequiredEdges > 0) {
                input.nextLine();
                for (int i = 0; i < nbNotRequiredEdges; ++i) {
                    splitted = input.nextLine().split(" ");
                    int firstNode = Integer.parseInt(splitted[2].substring(0, splitted[2].length() - 1));
                    int secondNode = Integer.parseInt(splitted[3].substring(0, splitted[3].length() - 1));
                    int cost = Integer.parseInt(splitted[7]);
                    nodeNeighborsData[firstNode - 1][secondNode - 1] = cost;
                    nodeNeighborsData[secondNode - 1][firstNode - 1] = cost;
                }
            }
            splitted = input.nextLine().split(":");
            depotNode = Integer.parseInt(splitted[1].trim());
            nbRequiredNodes = requiredNodes.size();
            int[] shortestPath = new int[nbNodes];
            findRequiredPaths(shortestPath);
            findDistanceBetweenEdges();
            findDistanceToDepot();
            findDistanceFromDepot(shortestPath);
        }
    }

    // Finds the shortest path from one node "origin" to all the other nodes of the
    // graph
    // thanks to the Dijkstra's algorithm
    public int minDistance(int[] shortestPath, boolean[] sptSet) {
        int min = Integer.MAX_VALUE, minIndex = -1;
        for (int i = 0; i < nbNodes; ++i)
            if (sptSet[i] == false && shortestPath[i] <= min) {
                min = shortestPath[i];
                minIndex = i;
            }
        return minIndex;
    }

    public void shortestPathFinder(int origin, int[] shortestPath) {
        int[] currentNeighbors;
        int currentNode, distance;
        boolean[] sptSet = new boolean[nbNodes];
        for (int i = 0; i < nbNodes; ++i) {
            shortestPath[i] = Integer.MAX_VALUE;
            sptSet[i] = false;
        }
        shortestPath[origin - 1] = 0;
        for (int count = 0; count < nbNodes; ++count) {
            currentNode = minDistance(shortestPath, sptSet);
            sptSet[currentNode] = true;
            currentNeighbors = nodeNeighborsData[currentNode];
            for (int neighbor = 0; neighbor < nbNodes; ++neighbor) {
                if (currentNeighbors[neighbor] != 0) {
                    distance = currentNeighbors[neighbor];
                    if (!sptSet[neighbor] && shortestPath[currentNode] + distance < shortestPath[neighbor]) {
                        shortestPath[neighbor] = shortestPath[currentNode] + distance;
                    }
                }
            }
        }
    }

    public void findRequiredPaths(int[] shortestPath) {
        nodesDistData = new int[nbRequiredNodes][];
        for (int i = 0; i < nbRequiredNodes; ++i) {
            nodesDistData[i] = new int[nbNodes];
        }
        for (int i = 0; i < nbRequiredNodes; ++i) {
            int node = requiredNodes.get(i);
            shortestPathFinder(node, shortestPath);
            for (int j = 0; j < nbNodes; ++j) {
                nodesDistData[i][j] = shortestPath[j];
            }
        }
    }

    public void findDistanceBetweenEdges() {
        edgesDistData = new int[2 * nbRequiredEdges][];
        for (int i = 0; i < 2 * nbRequiredEdges; ++i) {
            edgesDistData[i] = new int[2 * nbRequiredEdges];
        }
        for (int i = 0; i < 2 * nbRequiredEdges; ++i) {
            for (int j = 0; j < 2 * nbRequiredEdges; ++j) {
                if (destinationsData[i] == originsData[j]) {
                    edgesDistData[i][j] = 0;
                } else {
                    for (int k = 0; k < nbRequiredNodes; ++k) {
                        if (requiredNodes.get(k) == destinationsData[i]) {
                            edgesDistData[i][j] = nodesDistData[k][originsData[j] - 1];
                        }
                    }
                }
            }
        }
    }

    public void findDistanceToDepot() {
        distToDepotData = new int[2 * nbRequiredEdges];
        for (int i = 0; i < 2 * nbRequiredEdges; ++i) {
            if (destinationsData[i] == depotNode) {
                distToDepotData[i] = 0;
            } else {
                for (int k = 0; k < nbRequiredNodes; ++k) {
                    if (requiredNodes.get(k) == destinationsData[i]) {
                        distToDepotData[i] = nodesDistData[k][depotNode - 1];
                    }
                }
            }
        }
    }

    public void findDistanceFromDepot(int[] shortestPath) {
        distFromDepotData = new int[2 * nbRequiredEdges];
        for (int i = 0; i < 2 * nbRequiredEdges; ++i) {
            if (depotNode == originsData[i]) {
                distFromDepotData[i] = 0;
            } else {
                boolean depotIsRequiredNode = false;
                for (int k = 0; k < nbRequiredNodes; ++k) {
                    if (requiredNodes.get(k) == depotNode) {
                        depotIsRequiredNode = true;
                        distFromDepotData[i] = nodesDistData[k][originsData[i] - 1];
                    }
                }
                if (!depotIsRequiredNode) {
                    shortestPathFinder(depotNode, shortestPath);
                    distFromDepotData[i] = shortestPath[originsData[i] - 1];
                }
            }
        }
    }

    public static void main(String[] args) {
        if (args.length < 1) {
            System.err.println("Usage: java CapacitatedArcRouting inputFile [outputFile] [timeLimit]");
            System.exit(1);
        }
        try (HexalyOptimizer optimizer = new HexalyOptimizer()) {
            String instanceFile = args[0];
            String outputFile = args.length > 1 ? args[1] : null;
            String strTimeLimit = args.length > 2 ? args[2] : "20";

            CapacitatedArcRouting model = new CapacitatedArcRouting(optimizer);
            model.readInstance(instanceFile);
            model.solve(Integer.parseInt(strTimeLimit));
            if (outputFile != null) {
                model.WriteSolution(outputFile);
            }
        } catch (Exception ex) {
            System.err.println(ex);
            ex.printStackTrace();
            System.exit(1);
        }
    }
}
