import java.util.*;
import java.io.*;
import com.hexaly.optimizer.*;

public class CarSequencing {
    // Number of vehicles
    private int nbPositions;

    // Number of options
    private int nbOptions;

    // Number of classes
    private int nbClasses;

    // Options properties
    private int[] maxCarsPerWindow;
    private int[] windowSize;

    // Classes properties
    private int[] nbCars;
    private int[][] options;

    // Initial sequence
    private int[] initialSequence;

    // Hexaly Optimizer
    private final HexalyOptimizer optimizer;

    // LS Program variable
    private HxExpression sequence;

    // Objective
    private HxExpression totalViolations;

    private CarSequencing(HexalyOptimizer optimizer) {
        this.optimizer = optimizer;
    }

    /* Read instance data */
    private void readInstance(String fileName) throws IOException {
        try (Scanner input = new Scanner(new File(fileName))) {
            input.useLocale(Locale.ROOT);
            nbPositions = input.nextInt();
            nbOptions = input.nextInt();
            nbClasses = input.nextInt();

            maxCarsPerWindow = new int[nbOptions];
            for (int o = 0; o < nbOptions; ++o) {
                maxCarsPerWindow[o] = input.nextInt();
            }

            windowSize = new int[nbOptions];
            for (int o = 0; o < nbOptions; ++o) {
                windowSize[o] = input.nextInt();
            }

            options = new int[nbClasses][nbOptions];
            nbCars = new int[nbClasses];
            initialSequence = new int[nbPositions];
            int pos = 0;
            for (int c = 0; c < nbClasses; ++c) {
                input.nextInt(); // Note: index of class is read but not used
                nbCars[c] = input.nextInt();
                for (int o = 0; o < nbOptions; ++o) {
                    int v = input.nextInt();
                    options[c][o] = (v == 1) ? 1 : 0;
                }
                for (int p = pos; p < pos + nbCars[c]; ++p)
                    initialSequence[p] = c;
                pos += nbCars[c];
            }
        }
    }

    private void solve(int limit) {
        // Declare the optimization model
        HxModel model = optimizer.getModel();

        // sequence[i] = j if class initially planned on position j is produced on position i
        sequence = model.listVar(nbPositions);

        // Create HexalyOptimizer arrays to be able to access them with "at" operators
        HxExpression initialArray = model.array(initialSequence);
        HxExpression optionArray = model.array(options);

        // Number of cars with option o in each window
        HxExpression[][] nbCarsWindows = new HxExpression[nbOptions][];
        for (int o = 0; o < nbOptions; ++o) {
            HxExpression oExpr = model.createConstant(o);
            nbCarsWindows[o] = new HxExpression[nbPositions - windowSize[o] + 1];
            for (int j = 0; j < nbPositions - windowSize[o] + 1; ++j) {
                nbCarsWindows[o][j] = model.sum();
                for (int k = 0; k < windowSize[o]; ++k) {
                    HxExpression classAtPosition = model.at(initialArray, model.at(sequence, j + k));
                    nbCarsWindows[o][j].addOperand(model.at(optionArray, classAtPosition, oExpr));
                }
            }
        }

        // Number of violations of option o capacity in each window
        HxExpression[][] nbViolationsWindows = new HxExpression[nbOptions][];
        for (int o = 0; o < nbOptions; ++o) {
            nbViolationsWindows[o] = new HxExpression[nbPositions - windowSize[o] + 1];
            for (int j = 0; j < nbPositions - windowSize[o] + 1; ++j) {
                HxExpression delta = model.sub(nbCarsWindows[o][j], maxCarsPerWindow[o]);
                nbViolationsWindows[o][j] = model.max(0, delta);
            }
        }

        // Minimize the sum of violations for all options and all windows
        totalViolations = model.sum();
        for (int o = 0; o < nbOptions; ++o) {
            totalViolations.addOperands(nbViolationsWindows[o]);
        }

        model.minimize(totalViolations);
        model.close();

        // Set the initial solution
        sequence.getCollectionValue().clear();
        for (int p = 0; p < nbPositions; ++p)
            sequence.getCollectionValue().add(p);

        // Parametrize the optimizer
        optimizer.getParam().setTimeLimit(limit);

        optimizer.solve();
    }

    /*
     * Write the solution in a file with the following format:
     * - 1st line: value of the objective;
     * - 2nd line: for each position p, index of class at positions p.
     */
    private void writeSolution(String fileName) throws IOException {
        try (PrintWriter output = new PrintWriter(fileName)) {
            output.println(totalViolations.getValue());
            for (int p = 0; p < nbPositions; ++p) {
                output.print(initialSequence[(int) sequence.getCollectionValue().get(p)] + " ");
            }
            output.println();
        }
    }

    public static void main(String[] args) {
        if (args.length < 1) {
            System.err.println("Usage: java CarSequencing inputFile [outputFile] [timeLimit]");
            System.exit(1);
        }

        String instanceFile = args[0];
        String outputFile = args.length > 1 ? args[1] : null;
        String strTimeLimit = args.length > 2 ? args[2] : "60";

        try (HexalyOptimizer optimizer = new HexalyOptimizer()) {
            CarSequencing model = new CarSequencing(optimizer);
            model.readInstance(instanceFile);
            model.solve(Integer.parseInt(strTimeLimit));
            if (outputFile != null) {
                model.writeSolution(outputFile);
            }
        } catch (Exception ex) {
            System.err.println(ex);
            ex.printStackTrace();
            System.exit(1);
        }
    }
}
