import hexaly.optimizer
import sys


def read_integers(filename):
    with open(filename) as f:
        return [int(elem) for elem in f.read().split()]

#
# Read instance data
#


def read_instance(instance_file):
    file_it = iter(read_integers(instance_file))
    nb_positions = next(file_it)
    nb_options = next(file_it)
    nb_classes = next(file_it)
    max_cars_per_window = [next(file_it) for i in range(nb_options)]
    window_size = [next(file_it) for i in range(nb_options)]
    nb_cars = []
    options = []
    initial_sequence = []

    for c in range(nb_classes):
        next(file_it)  # Note: index of class is read but not used
        nb_cars.append(next(file_it))
        options.append([next(file_it) == 1 for i in range(nb_options)])
        [initial_sequence.append(c) for p in range(nb_cars[c])]

    return nb_positions, nb_options, max_cars_per_window, window_size, options, \
        initial_sequence


def main(instance_file, output_file, time_limit):
    nb_positions, nb_options, max_cars_per_window, window_size, options, \
        initial_sequence = read_instance(instance_file)

    with hexaly.optimizer.HexalyOptimizer() as optimizer:
        #
        # Declare the optimization model
        #
        model = optimizer.model

        # sequence[i] = j if class initially planned on position j is produced on position i
        sequence = model.list(nb_positions)

        # sequence is a permutation of the initial production plan, all indexes must
        # appear exactly once
        model.constraint(model.partition(sequence))

        # Create Hexaly arrays to be able to access them with "at" operators
        initial_array = model.array(initial_sequence)
        option_array = model.array(options)

        # Number of cars with option o in each window
        nb_cars_windows = [None] * nb_options
        for o in range(nb_options):
            nb_cars_windows[o] = [None] * nb_positions
            for j in range(nb_positions - window_size[o] + 1):
                nb_cars_windows[o][j] = model.sum()
                for k in range(window_size[o]):
                    class_at_position = initial_array[sequence[j + k]]
                    nb_cars_windows[o][j].add_operand(model.at(
                        option_array,
                        class_at_position,
                        o))

        # Number of violations of option o capacity in each window
        nb_violations_windows = [None] * nb_options
        for o in range(nb_options):
            nb_violations_windows[o] = [None] * nb_positions
            for p in range(nb_positions - window_size[o] + 1):
                nb_violations_windows[o][p] = model.max(
                    nb_cars_windows[o][p] - max_cars_per_window[o], 0)

        # Minimize the sum of violations for all options and all windows
        total_violations = model.sum(
            nb_violations_windows[o][p]
            for p in range(nb_positions - window_size[o] + 1) for o in range(nb_options))
        model.minimize(total_violations)

        model.close()

        # Set the initial solution
        sequence.get_value().clear()
        for p in range(nb_positions):
            sequence.get_value().add(p)

        # Parameterize the optimizer
        optimizer.param.time_limit = time_limit

        optimizer.solve()

        #
        # Write the solution in a file with the following format:
        # - 1st line: value of the objective;
        # - 2nd line: for each position p, index of class at positions p.
        #
        if output_file is not None:
            with open(output_file, 'w') as f:
                f.write("%d\n" % total_violations.value)
                for p in range(nb_positions):
                    f.write("%d " % initial_sequence[sequence.value[p]])

                f.write("\n")


if __name__ == '__main__':
    if len(sys.argv) < 2:
        print("Usage: python car_sequencing.py instance_file [output_file] [time_limit]")
        sys.exit(1)

    instance_file = sys.argv[1]
    output_file = sys.argv[2] if len(sys.argv) >= 3 else None
    time_limit = int(sys.argv[3]) if len(sys.argv) >= 4 else 60
    main(instance_file, output_file, time_limit)
