using System;
using System.IO;
using System.Collections.Generic;
using Hexaly.Optimizer;

public class ClusteredVehicleRouting : IDisposable
{
    // Hexaly Optimizer
    HexalyOptimizer optimizer;

    // Number of customers
    int nbCustomers;

    // Capacity of the trucks
    int truckCapacity;

    // Demand on each customer
    long[] demandsData;

    // Customers in each cluster 
    long[][] clustersData;

    // Distance matrix between customers
    long[][] distMatrixData;

    // Distances between customers and depot
    long[] distDepotData;

    // Number of trucks
    int nbTrucks;

    // Number of Clusters
    int nbClusters;

    // Decision variables
    HxExpression[] truckSequences;
    HxExpression[] clustersSequences;


    // Distance traveled by all the trucks
    HxExpression totalDistance;

    public ClusteredVehicleRouting()
    {
        optimizer = new HexalyOptimizer();
    }

    /* Read instance data */
    void ReadInstance(string fileName)
    {
        ReadInputCluCvrp(fileName);
    }

    public void Dispose()
    {
        if (optimizer != null)
            optimizer.Dispose();
    }

    void Solve(int limit)
    {
        // Declare the optimization model
        HxModel model = optimizer.GetModel();

        HxExpression[] clustersDistances = new HxExpression[nbClusters];
        HxExpression[] routeDistances = new HxExpression[nbTrucks];
        HxExpression[] initialNodes = new HxExpression[nbClusters];
        HxExpression[] endNodes = new HxExpression[nbClusters];

        clustersSequences = new HxExpression[nbClusters];
        // A list is created for each cluster, to determine the order within the cluster
        for ( int k = 0; k < nbClusters; ++k)
        {
            int c = (int) clustersData[k].Length;
            clustersSequences[k] = model.List(c);
            // All customers in the cluster must be visited 
            model.Constraint(model.Count(clustersSequences[k]) == c);
        }

        // Create HexalyOptimizer arrays to be able to access them with an "at" operator
        HxExpression demands = model.Array(demandsData);
        HxExpression distDepot = model.Array(distDepotData);
        HxExpression distMatrix = model.Array(distMatrixData);
        HxExpression clusters = model.Array(clustersData);
        
        for (int k = 0; k < nbClusters; ++k)
        {
            HxExpression sequence = clustersSequences[k];
            HxExpression c = model.Count(sequence);

            HxExpression routeDistances_lambda = model.LambdaFunction(
                i => model.At(distMatrix,
                     clusters[k][sequence[i - 1]], clusters[k][sequence[i]]) );
            
            // Distance traveled within cluster k
            clustersDistances[k] = model.Sum(model.Range(1,c), routeDistances_lambda);

            // first and last point visited when traveled into cluster k
            initialNodes[k] = clusters[k][sequence[0]];
            endNodes[k] = clusters[k][sequence[c - 1]];
        }

        truckSequences = new HxExpression[nbTrucks];
        // Sequence of clusters visited by each truck
        for (int k = 0; k < nbTrucks; ++k)
            truckSequences[k] = model.List(nbClusters);

        // All customers must be visited by the trucks
        model.Constraint(model.Partition(truckSequences));

        HxExpression valueDistClusters = model.Array(clustersDistances);
        HxExpression initials = model.Array(initialNodes);
        HxExpression ends = model.Array(endNodes);
        for (int k = 0; k < nbTrucks; ++k)
        {
            HxExpression sequence = truckSequences[k];
            HxExpression c = model.Count(sequence);

            // The quantity needed in each route must not exceed the truck capacity
            HxExpression demandLambda = model.LambdaFunction(j => demands[j]);
            HxExpression routeQuantity = model.Sum(sequence, demandLambda);
            model.Constraint(routeQuantity <= truckCapacity);

            // Distance traveled by truck k
            // = distance in each cluster + distance between clusters + distance with depot 
            // at the beginning end at the end of a route
            HxExpression routeDistances_lambda = model.LambdaFunction(
                    i => model.Sum( model.At(valueDistClusters, sequence[i]),
                    model.At(distMatrix, ends[sequence[i - 1]], initials[sequence[i]])));
            routeDistances[k] = model.Sum(model.Range(1, c), routeDistances_lambda)
                    + model.If(c > 0, valueDistClusters[ model.At(sequence,0)]
                    + distDepot[initials[sequence[0]]] + distDepot[ends[sequence[c - 1]]], 0);
        }

        totalDistance = model.Sum(routeDistances);

        // Objective:  minimize the distance traveled
        model.Minimize(totalDistance);

        model.Close();

        // Parametrize the optimizer
        optimizer.GetParam().SetTimeLimit(limit);

        optimizer.Solve();
    }

    /* Write the solution in a file with the following format:
     * -  total distance
     * - for each truck the customers visited (omitting the start/end at the depot) */
    void WriteSolution(string fileName)
    {
        using (StreamWriter output = new StreamWriter(fileName))
        {
            output.WriteLine(totalDistance.GetValue());
            for (int k = 0; k < nbTrucks; ++k)
            {
                // Values in sequence are in [0..nbCustomers-1]. +2 is to put it back in 
                // [2..nbCustomers+1] as in the data files (1 being the depot)
                HxCollection customersCollection = truckSequences[k].GetCollectionValue();
                for (int i = 0; i < customersCollection.Count(); ++i)
                {
                    long cluster = customersCollection[i];
                    HxCollection clustersCollection = 
                        clustersSequences[cluster].GetCollectionValue();
                    for (int j = 0; j < clustersCollection.Count(); ++j)
                        output.Write((clustersData[cluster][clustersCollection[j]] + 2) + " ");
                }
                output.WriteLine();
            }
        }
    }

    public static void Main(string[] args)
    {
        if (args.Length < 1)
        {
            Console.WriteLine("Usage: ClusteredVehicleRouting.cs inputFile [solFile] [timeLimit]");
            Environment.Exit(1);
        }
        string instanceFile = args[0];
        string outputFile = args.Length > 1 ? args[1] : null;
        string strTimeLimit = args.Length > 2 ? args[2] : "5";
        string strNbTrucks = args.Length > 3 ? args[3] : "0";

        using (ClusteredVehicleRouting model = new ClusteredVehicleRouting())
        {
            model.ReadInstance(instanceFile);
            model.Solve(int.Parse(strTimeLimit));
            if (outputFile != null)
                model.WriteSolution(outputFile);
        }
    }

    // The input files follow the "Augerat" format
    private void ReadInputCluCvrp(string fileName)
    {
        using (StreamReader input = new StreamReader(fileName))
        {   
            int nbNodes = 0;
            string[] splitted;
            while (true) 
            {
                splitted = input.ReadLine().Split(':');
                if (splitted[0].Contains("DIMENSION"))
                {
                    nbNodes = int.Parse(splitted[1]);
                    nbCustomers = nbNodes - 1;
                }
                else if (splitted[0].Contains("VEHICLES"))
                    nbTrucks = int.Parse(splitted[1]);
                else if (splitted[0].Contains("GVRP_SETS"))
                    nbClusters = int.Parse(splitted[1]);
                else if (splitted[0].Contains("CAPACITY"))
                    truckCapacity = int.Parse(splitted[1]);
                else if (splitted[0].Contains("NODE_COORD_SECTION"))
                    break;
            }
            int[] customersX = new int[nbCustomers];
            int[] customersY = new int[nbCustomers];
            int depotX = 0,
                depotY = 0;
            char[] delimiterChars = { ' ', '.' };
            for (int n = 1; n <= nbNodes; ++n)
            {
                splitted = input.ReadLine().Split(delimiterChars);
                int id = int.Parse(splitted[0]);
                if ( id != n)
                    throw new Exception("Unexpected index");
                if (n == 1)
                {
                    depotX = int.Parse(splitted[1]);
                    depotY = int.Parse(splitted[3]);
                } 
                else
                {
                    // -2 because original customer indices are in 2..nbNodes
                    customersX[n - 2] = int.Parse(splitted[1]);
                    customersY[n - 2] = int.Parse(splitted[3]);
                }
            }

            ComputeDistanceMatrix(depotX, depotY, customersX, customersY);

            splitted = input.ReadLine().Split(' ');
            if (!splitted[0].Contains("GVRP_SET_SECTION"))
                throw new Exception("Expected keyword GVRP_SET_SECTION");

            clustersData = new long[nbClusters][];
            for (int n = 1; n <= nbClusters; ++n)
            {
                splitted = input
                    .ReadLine()
                    .Split((char[])null, StringSplitOptions.RemoveEmptyEntries);
                if (int.Parse(splitted[0]) != n)
                    throw new Exception("Unexpected index");
                List<long> cluster = new List<long>();
                int i = 1;
                var customer = int.Parse(splitted[1]);
                while (customer != -1)
                {
                    // -2 because original customer indices are in 2..nbNodes
                    cluster.Add(customer - 2);
                    i++;
                    customer = int.Parse(splitted[i]);
                }
                clustersData[n - 1] = cluster.ToArray();
            }    
            
            splitted = input.ReadLine().Split(' ');
            if (!splitted[1].Contains("DEMAND_SECTION"))
                throw new Exception("Expected keyword DEMAND_SECTION");

            demandsData = new long[nbCustomers];
            for (int n = 1; n <= nbClusters; ++n)
            {
                splitted = input
                    .ReadLine()
                    .Split((char[])null, StringSplitOptions.RemoveEmptyEntries);
                if (int.Parse(splitted[0]) != n)
                    throw new Exception("Unexpected index");
                var demand = int.Parse(splitted[1]);
                demandsData[n - 1] = demand;
            }

        }
    }

    // Compute the distance matrix
    private void ComputeDistanceMatrix(int depotX, int depotY, int[] customersX, int[] customersY)
    {
        distMatrixData = new long[nbCustomers][];
        for (int i = 0; i < nbCustomers; ++i)
            distMatrixData[i] = new long[nbCustomers];

        for (int i = 0; i < nbCustomers; ++i)
        {
            distMatrixData[i][i] = 0;
            for (int j = i + 1; j < nbCustomers; ++j)
            {
                long dist = ComputeDist(customersX[i], customersX[j], customersY[i], 
                    customersY[j]);
                distMatrixData[i][j] = dist;
                distMatrixData[j][i] = dist;
            }
        }

        distDepotData = new long[nbCustomers];
        for (int i = 0; i < nbCustomers; ++i)
            distDepotData[i] = ComputeDist(depotX, customersX[i], depotY, customersY[i]);
    }

    private long ComputeDist(int xi, int xj, int yi, int yj)
    {
        double exactDist = Math.Sqrt(Math.Pow(xi - xj, 2) + Math.Pow(yi - yj, 2));
        return Convert.ToInt64(Math.Round(exactDist));
    }
}
