import java.util.*;
import java.io.*;
import com.hexaly.optimizer.*;

public class ClusteredVehicleRouting {
    // Hexaly Optimizer
    private final HexalyOptimizer optimizer;

    // Number of customers
    int nbCustomers;

    // Capacity of the trucks
    private int truckCapacity;

    // Demand on each customer
    private long[] demandsData;

    // Customers in each cluster 
    private long[][] clustersData;

    // Distance matrix between customers
    private long[][] distMatrixData;

    // Distances between customers and depot
    private long[] distDepotData;

    // Number of trucks
    private int nbTrucks;

    // Number of clusters
    private int nbClusters;

    // Decision variables
    private HxExpression[] truckSequences;
    private HxExpression[] clustersSequences;

    // Distance traveled by all the trucks
    private HxExpression totalDistance;

    private ClusteredVehicleRouting(HexalyOptimizer optimizer) {
        this.optimizer = optimizer;
    }

    private void solve(int limit) {
        // Declare the optimization model
        HxModel model = optimizer.getModel();

        // Create HexalyOptimizer arrays to be able to access them with an "at" operator
        HxExpression demands = model.array(demandsData);
        HxExpression distDepot = model.array(distDepotData);
        HxExpression distMatrix = model.array(distMatrixData);
        HxExpression[] distClusters = new HxExpression[nbClusters];
        HxExpression[] distRoutes = new HxExpression[nbTrucks];
        HxExpression[] initialNodes = new HxExpression[nbClusters];
        HxExpression[] endNodes = new HxExpression[nbClusters];

        clustersSequences  = new HxExpression[nbClusters];
        // A list is created for each cluster, to determine the order within the cluster
        for (int k = 0; k < nbClusters; ++k) {
            clustersSequences[k] = model.listVar(clustersData[k].length);
            // All customers in the cluster must be visited 
            model.constraint(model.eq(model.count(clustersSequences[k]), 
                clustersData[k].length));
        }

        for (int k = 0; k < nbClusters; ++k) {
            HxExpression sequence = clustersSequences[k];
            HxExpression sequenceCluster = model.array(clustersData[k]);
            HxExpression c = model.count(sequence);

            // Distance traveled within cluster k
            HxExpression distLambda = model
                .lambdaFunction(i -> model.at(distMatrix, 
                    model.at(sequenceCluster, model.at(sequence, model.sub(i, 1))), 
                    model.at(sequenceCluster, model.at(sequence, i) ))
                );
            distClusters[k] = model.sum(model.range(1,c), distLambda);

            // First and last point when visiting cluster k
            initialNodes[k] = model.at(sequenceCluster, model.at(sequence, 0));
            endNodes[k]     = model.at(sequenceCluster, model.at(sequence, model.sub(c,1)));
        }

        truckSequences = new HxExpression[nbTrucks];
        // Sequence of clusters visited by each truck.
        for (int k = 0; k < nbTrucks; ++k)
            truckSequences[k] = model.listVar(nbClusters);
        // Each cluster must be visited by exactly one truck
        model.constraint(model.partition(truckSequences));

        HxExpression valueDistCluster = model.array(distClusters);
        HxExpression initials         = model.array(initialNodes);
        HxExpression ends             = model.array(endNodes);
        
        for (int k = 0; k < nbTrucks; ++k) {
            HxExpression sequence = truckSequences[k];
            HxExpression c = model.count(sequence);

            // The quantity needed in each route must not exceed the truck capacity
            HxExpression demandLambda = model.lambdaFunction(j -> model.at(demands, j));
            HxExpression routeQuantity = model.sum(sequence, demandLambda);
            model.constraint(model.leq(routeQuantity, truckCapacity));

            // Distance traveled by truck k
            // = distance in each cluster + distance between clusters + distance with depot 
            // at the beginning end at the end of a route
            HxExpression distLambda = model.lambdaFunction(i -> model.sum(
                    model.at(valueDistCluster, model.at(sequence,i)),  
                    model.at(distMatrix, model.at(ends, model.at(sequence, model.sub(i, 1))), 
                    model.at(initials, model.at(sequence, i)))) );
            distRoutes[k] = model.sum(
                    model.sum(model.range(1, c), distLambda),
                    model.iif(model.gt(c, 0), model.sum( 
                    model.at(valueDistCluster, model.at(sequence,0)),
                    model.at(distDepot, model.at(initials, model.at(sequence, 0))),
                    model.at(distDepot, model.at(ends, model.at(sequence, model.sub(c, 1))))),
                    0) );
        }

        totalDistance = model.sum(distRoutes);

        // Objective: minimize the distance traveled
        model.minimize(totalDistance);

        model.close();

        // Parametrize the optimizer
        optimizer.getParam().setTimeLimit(limit);

        optimizer.solve();
    }

    /*
     * Write the solution in a file with the following format:
     * - number of trucks used and total distance
     * - for each truck the customers visited (omitting the start/end at the depot)
     */
    private void writeSolution(String fileName) throws IOException {
        try (PrintWriter output = new PrintWriter(fileName)) {
            output.println( totalDistance.getValue());
            for (int k = 0; k < nbTrucks; ++k) {

                // Values in sequence are in [0..nbCustomers-1]. +2 is to put it back in
                // [2..nbCustomers+1] as in the data files (1 being the depot)
                HxCollection customersCollection = truckSequences[k].getCollectionValue();
                for (int i = 0; i < customersCollection.count(); ++i) {
                    int cluster = (int) customersCollection.get(i);
                    HxCollection clustersCollection = 
                            clustersSequences[cluster].getCollectionValue();
                    for (int j = 0; j < clustersCollection.count(); ++j)
                        output.print((clustersData[cluster][(int) clustersCollection.get(j)] + 2)
                            + " ");
                }
                output.println();
            }
        }
    }

    // The input files follow the "Augerat" format
    private void readInstance(String fileName) throws IOException {
        try (Scanner input = new Scanner(new File(fileName))) {
            input.useLocale(Locale.ROOT);
            int nbNodes = 0;
            String[] splitted;
            while (true) {
                splitted = input.nextLine().split(":");
                if (splitted[0].contains("DIMENSION")) {
                    nbNodes = Integer.parseInt(splitted[1].trim());
                    nbCustomers = nbNodes - 1;
                } else if (splitted[0].contains("VEHICLES")) {
                    nbTrucks = Integer.parseInt(splitted[1].trim());
                } else if (splitted[0].contains("GVRP_SETS")) {
                    nbClusters = Integer.parseInt(splitted[1].trim());
                } else if (splitted[0].contains("CAPACITY")) {
                    truckCapacity = Integer.parseInt(splitted[1].trim());
                } else if (splitted[0].contains("NODE_COORD_SECTION")) {
                    break;
                }
            }
            int[] customersX = new int[nbCustomers];
            int[] customersY = new int[nbCustomers];
            int depotX = 0, depotY = 0;
            for (int n = 1; n <= nbNodes; ++n) {
                int id = input.nextInt();
                if (id != n)
                    throw new IOException("Unexpected index");
                if (n == 1) {
                    depotX = (int) input.nextFloat();
                    depotY = (int) input.nextFloat();
                } else {
                    // -2 because original customer indices are in 2..nbNodes
                    customersX[n - 2] = (int) input.nextFloat();
                    customersY[n - 2] = (int) input.nextFloat();
                }
            }
            computeDistanceMatrix(depotX, depotY, customersX, customersY);

            input.nextLine().split(""); // End the last line
            String name = input.nextLine();
            if (!name.contains("GVRP_SET_SECTION")) {
                throw new RuntimeException("Expected keyword GVRP_SET_SECTION");
            }
            clustersData = new long[nbClusters][];
            for (int n = 1; n <= nbClusters; ++n) {
                List<Long> cluster = new ArrayList<>();
                int id = input.nextInt();
                if (id != n)
                    throw new IOException("Unexpected index");
                long customer = input.nextInt();
                while (customer != -1) {
                    // -2 because original customer indices are in 2..nbNodes
                    cluster.add(customer - 2);
                    customer = input.nextInt();
                }
                long[] clusterArray = cluster.stream()
                        .mapToLong(Long::intValue).toArray();
                clustersData[n - 1] = clusterArray;
            }

            input.nextLine().split(""); 
            name = input.nextLine();
            if (!name.contains("DEMAND_SECTION")) {
                throw new RuntimeException("Expected keyword DEMAND_SECTION");
            }
            demandsData = new long[nbCustomers];
            for (int n = 1; n <= nbClusters; ++n) {
                int id = input.nextInt();
                if (id != n) throw new IOException("Unexpected index");
                int demand = input.nextInt();
                demandsData[n - 1] = demand;
            }
        }
    }

    // Compute the distance matrix
    private void computeDistanceMatrix(int depotX, int depotY, int[] customersX, int[] customersY) {
        distMatrixData = new long[nbCustomers][nbCustomers];
        for (int i = 0; i < nbCustomers; ++i) {
            distMatrixData[i][i] = 0;
            for (int j = i + 1; j < nbCustomers; ++j) {
                long dist = computeDist(customersX[i], customersX[j], customersY[i], customersY[j]);
                distMatrixData[i][j] = dist;
                distMatrixData[j][i] = dist;
            }
        }

        distDepotData = new long[nbCustomers];
        for (int i = 0; i < nbCustomers; ++i) {
            distDepotData[i] = computeDist(depotX, customersX[i], depotY, customersY[i]);
        }
    }

    private long computeDist(int xi, int xj, int yi, int yj) {
        double exactDist = Math.sqrt(Math.pow(xi - xj, 2) + Math.pow(yi - yj, 2));
        return Math.round(exactDist);
    }

    public static void main(String[] args) {
        if (args.length < 1) {
            System.err.println("Usage: java ClusteredVehicleRouting inputFile [outputFile] [timeLimit]");
            System.exit(1);
        }
        try (HexalyOptimizer optimizer = new HexalyOptimizer()) {
            String instanceFile = args[0];
            String outputFile = args.length > 1 ? args[1] : null;
            String strTimeLimit = args.length > 2 ? args[2] : "20";

            ClusteredVehicleRouting model = new ClusteredVehicleRouting(optimizer);
            model.readInstance(instanceFile);
            model.solve(Integer.parseInt(strTimeLimit));
            if (outputFile != null) {
                model.writeSolution(outputFile);
            }
        } catch (Exception ex) {
            System.err.println(ex);
            ex.printStackTrace();
            System.exit(1);
        }
    }
}
