using System;
using System.IO;
using System.Collections.Generic;

using Hexaly.Optimizer;
using Newtonsoft.Json.Linq;
using Newtonsoft.Json;

public class Darp : IDisposable
{
    HexalyOptimizer optimizer;

    int nbClients;
    int nbNodes;
    int nbVehicles;
    double depotTwEnd;
    int capacity;
    double scale;
    double factor;

    int[] quantitiesData;
    double[] startsData;
    double[] endsData;
    double[] loadingTimesData;
    double[] maxTravelTimes;

    double[][] distances;
    double[] distanceWarehouseData;
    double[] timeWarehouseData;
    double[][] distanceMatrixData;
    double[][] timeMatrixData;

    HxExpression[] routes;
    HxExpression[] depotStarts;
    HxExpression[] waiting;

    HxExpression[] clientLateness;
    HxExpression[] latenessPlusHomeLateness;

    HxExpression totalLateness;
    HxExpression totalClientLateness;
    HxExpression totalDistance;

    public Darp()
    {
        optimizer = new HexalyOptimizer();
    }

    /* Read instance data */
    void ReadInstance(string fileName)
    {
        JObject instance = JObject.Parse(File.ReadAllText(fileName));

        nbClients = (int)instance["nbClients"];
        nbNodes = (int)instance["nbNodes"];
        nbVehicles = (int)instance["nbVehicles"];
        depotTwEnd = (double)instance["depot"]["twEnd"];
        capacity = (int)instance["capacity"];
        scale = (double)instance["scale"];

        quantitiesData = new int[2 * nbClients];
        startsData = new double[2 * nbClients];
        endsData = new double[2 * nbClients];
        loadingTimesData = new double[2 * nbClients];
        maxTravelTimes = new double[2 * nbClients];

        for (int k = 0; k < nbClients; ++k)
        {
            quantitiesData[k] = (int)instance["clients"][k]["nbClients"];
            quantitiesData[k + nbClients] = -(int)instance["clients"][k]["nbClients"];

            startsData[k] = (double)instance["clients"][k]["pickup"]["start"];
            startsData[k + nbClients] = (double)instance["clients"][k]["delivery"]["start"];

            endsData[k] = (double)instance["clients"][k]["pickup"]["end"];
            endsData[k + nbClients] = (double)instance["clients"][k]["delivery"]["end"];

            loadingTimesData[k] = (double)instance["clients"][k]["pickup"]["loadingTime"];
            loadingTimesData[k + nbClients] = (double)instance["clients"][k]["delivery"]["loadingTime"];

            maxTravelTimes[k] = (double)instance["clients"][k]["pickup"]["maxTravelTime"];
            maxTravelTimes[k + nbClients] =
                (double)instance["clients"][k]["delivery"]["maxTravelTime"];
        }

        distances = new double[nbNodes + 1][];
        for (int i = 0; i < nbNodes + 1; ++i)
        {
            distances[i] = new double[nbNodes + 1];
            for (int j = 0; j < nbNodes + 1; ++j)
            {
                distances[i][j] = (double)instance["distanceMatrix"][i][j];
            }
        }

        factor = 1 / ((double)instance["scale"] * (double)instance["speed"]);

        distanceWarehouseData = new double[nbNodes];
        timeWarehouseData = new double[nbNodes];
        for (int k = 0; k < nbNodes; ++k)
        {
            distanceWarehouseData[k] = distances[0][k+1];
            timeWarehouseData[k] = distanceWarehouseData[k] * factor;
        }

        distanceMatrixData = new double[nbNodes][];
        timeMatrixData = new double[nbNodes][];
        for (int i = 0; i < nbNodes; ++i)
        {
            distanceMatrixData[i] = new double[nbNodes];
            timeMatrixData[i] = new double[nbNodes];
            for (int j = 0; j < nbNodes; ++j)
            {
                distanceMatrixData[i][j] = distances[i+1][j+1];
                timeMatrixData[i][j] = distanceMatrixData[i][j] * factor;
            }
        }

    }

    public void Dispose()
    {
        if (optimizer != null)
            optimizer.Dispose();
    }

    void Solve(int limit)
    {
        HxModel model = optimizer.GetModel();

        // routes[k] represents the nodes visited by vehicle k
        routes = new HxExpression[nbVehicles];
        depotStarts = new HxExpression[nbVehicles];
        for (int k = 0; k < nbVehicles; ++k)
        {
            routes[k] = model.List(nbNodes);
            depotStarts[k] = model.Float(0.0, depotTwEnd);
        }
        // waiting[k] is the waiting time at node k
        waiting = new HxExpression[nbNodes];
        for (int k = 0; k < nbNodes; ++k)
        {
            waiting[k] = model.Float(0.0, depotTwEnd);
        }
        // Each node is taken by one vehicle
        model.Constraint(model.Partition(routes));

        HxExpression quantities = model.Array(quantitiesData);
        HxExpression timeWarehouse = model.Array(timeWarehouseData);
        HxExpression loadingTimes = model.Array(loadingTimesData);
        HxExpression starts = model.Array(startsData);
        HxExpression ends = model.Array(endsData);
        HxExpression waitingArray = model.Array(waiting);
        HxExpression timeMatrix = model.Array();
        HxExpression distanceMatrix = model.Array();
        for (int i = 0; i < nbNodes; ++i)
        {
            timeMatrix.AddOperand(model.Array(timeMatrixData[i]));
            distanceMatrix.AddOperand(model.Array(distanceMatrixData[i]));
        }
        HxExpression distanceWarehouse = model.Array(distanceWarehouseData);

        HxExpression[] times = new HxExpression[nbVehicles];
        HxExpression[] lateness = new HxExpression[nbVehicles];
        HxExpression[] homeLateness = new HxExpression[nbVehicles];
        HxExpression[] routeDistances = new HxExpression[nbVehicles];

        for (int k = 0; k < nbVehicles; ++k)
        {
            HxExpression route = routes[k];
            HxExpression c = model.Count(route);

            HxExpression demandLambda = model.LambdaFunction(
                (i, prev) => prev + quantities[route[i]]
            );
            // routeQuantities[k][i] indicates the number of clients in vehicle k
            // at its i-th taken node
            HxExpression routeQuantities = model.Array(model.Range(0, c), demandLambda);
            HxExpression quantityLambda = model.LambdaFunction(i => routeQuantities[i] <= capacity);
            // Vehicles have a maximum capacity
            model.Constraint(model.And(model.Range(0, c), quantityLambda));

            HxExpression timesLambda = model.LambdaFunction(
                (i, prev) =>
                    model.Max(
                        starts[route[i]],
                        model.If(
                            i == 0,
                            depotStarts[k] + timeWarehouse[route[0]],
                            prev + timeMatrix[route[i-1]][route[i]]
                        )
                    ) + waitingArray[route[i]] + loadingTimes[route[i]]);
            // times[k][i] is the time at which vehicle k leaves the i-th node
            // (after waiting and loading time at node i)
            times[k] = model.Array(model.Range(0, c), timesLambda);

            HxExpression latenessLambda = model.LambdaFunction(
                i =>
                    model.Max(
                        0,
                        times[k][i] - loadingTimes[route[i]] - ends[route[i]]
                    )
            );
            // Total lateness of the k-th route
            lateness[k] = model.Sum(model.Range(0, c), latenessLambda);

            homeLateness[k] = model.If(
                c > 0,
                model.Max(0, times[k][c-1] + timeWarehouse[route[c-1]] - depotTwEnd),
                0
            );

            HxExpression routeDistLambda = model.LambdaFunction(
                i => distanceMatrix[route[i-1]][route[i]]
            );
            routeDistances[k] = model.Sum(
                model.Range(1, c),
                routeDistLambda
            ) + model.If(
                c > 0,
                distanceWarehouse[route[0]] + distanceWarehouse[route[c-1]],
                0
            );
        }
     
        HxExpression routesArray = model.Array(routes);
        HxExpression timesArray = model.Array(times);
        clientLateness = new HxExpression[nbClients];

        for (int k = 0; k < nbClients; ++k)
        {
            // For each pickup node k, its associated delivery node is k + nbClients
            HxExpression pickupListIndex = model.Find(routesArray, k);
            HxExpression deliveryListIndex = model.Find(routesArray, k + nbClients);
            // A client picked up in route i is delivered in route i
            model.Constraint(pickupListIndex == deliveryListIndex);

            HxExpression clientList = routesArray[pickupListIndex];
            HxExpression pickupIndex = model.IndexOf(clientList, k);
            HxExpression deliveryList = routesArray[deliveryListIndex];
            HxExpression deliveryIndex = model.IndexOf(deliveryList, k + nbClients);
            // Pickup before delivery
            model.Constraint(pickupIndex < deliveryIndex);

            HxExpression pickupTime = timesArray[pickupListIndex][pickupIndex];
            HxExpression deliveryTime =
                timesArray[deliveryListIndex][deliveryIndex] - loadingTimes[k + nbClients];
            HxExpression travelTime = deliveryTime - pickupTime;
            clientLateness[k] = model.Max(travelTime - maxTravelTimes[k], 0);
        }

        latenessPlusHomeLateness = new HxExpression[nbVehicles];
        for (int k = 0; k < nbVehicles; ++k)
        {
            latenessPlusHomeLateness[k] = lateness[k] + homeLateness[k];
        }
        totalLateness = model.Sum(latenessPlusHomeLateness);
        totalClientLateness = model.Sum(clientLateness);
        totalDistance = model.Sum(routeDistances);

        model.Minimize(totalLateness);
        model.Minimize(totalClientLateness);
        model.Minimize(totalDistance / scale);

        model.Close();

        optimizer.GetParam().SetTimeLimit(limit);

        optimizer.GetParam().SetSeed(5);
        optimizer.Solve();
    }

    /* Write the solution in a file with the following format:
     *  - total lateness on the routes, total client lateness, total distance
     *  - for each vehicle, the depot start time, the nodes visited (omitting the start/end at the
     * depot), and the waiting time at each node */
    void WriteSolution(string fileName)
    {
        using (StreamWriter output = new StreamWriter(fileName))
        {
            output.WriteLine(
                totalLateness.GetDoubleValue()
                + " "
                + totalClientLateness.GetDoubleValue()
                + " "
                + totalDistance.GetDoubleValue()
            );
            for (int k = 0; k < nbVehicles; ++k)
            {
                HxCollection route = routes[k].GetCollectionValue();
                output.Write("Vehicle " + (k + 1) + " (" + depotStarts[k].GetDoubleValue() + "): ");
                for (int i = 0; i < route.Count(); ++i)
                    output.Write(route[i] + " (" + (waiting[route[i]].GetDoubleValue() + "), "));
                output.WriteLine();
            }
        }
    }

    public static void Main(string[] args)
    {
        if (args.Length < 1)
        {
            Console.WriteLine("Usage: Darp inputFile [solFile] [timeLimit]");
            Environment.Exit(1);
        }
        string instanceFile = args[0];
        string outputFile = args.Length > 1 ? args[1] : null;
        string strTimeLimit = args.Length > 2 ? args[2] : "20";

        using (Darp model = new Darp())
        {
            model.ReadInstance(instanceFile);
            model.Solve(int.Parse(strTimeLimit));
            if (outputFile != null)
                model.WriteSolution(outputFile);
        }
    }
}