import java.util.*;
import java.io.*;
import com.hexaly.optimizer.*;

public class FlexibleJobshopSetup {
    // Number of jobs
    private int nbJobs;
    // Number of machines
    private int nbMachines;
    // Number of tasks
    private int nbTasks;
    // Processing time for each task, for each machine
    private long[][] taskProcessingTimeData;
    // Setup time between every two consecutive tasks, for each machine
    private int[][][] taskSetupTimeData;
    // For each job, for each operation, the corresponding task id
    private int[][] jobOperationTask;
    // Number of operations for each job;
    private int[] nbOperations;
    // Trivial upper bound for the end times of the tasks
    private long maxEnd;
    // Constant for incompatible machines
    private final int INFINITE = 1000000;

    // Hexaly Optimizer
    final HexalyOptimizer optimizer;
    // Decision variables: time range of each task
    private HxExpression[] tasks;
    // Decision variables: sequence of tasks on each machine
    private HxExpression[] jobsOrder;
    // For each task, the selected machine
    private HxExpression[] taskMachine;
    // Objective = minimize the makespan: end of the last task
    private HxExpression makespan;

    public FlexibleJobshopSetup(HexalyOptimizer optimizer) throws IOException {
        this.optimizer = optimizer;
    }

    public void readInstance(String fileName) throws IOException {
        try (Scanner input = new Scanner(new File(fileName))) {
            input.useLocale(Locale.ROOT);
            nbJobs = input.nextInt();
            nbMachines = input.nextInt();
            input.next(); // skip last number

            nbTasks = 0;
            long[][][] processingTime = new long[nbJobs][][];
            jobOperationTask = new int[nbJobs][];
            nbOperations = new int[nbJobs];
            for (int j = 0; j < nbJobs; ++j) {
                nbOperations[j] = input.nextInt();
                jobOperationTask[j] = new int[nbOperations[j]];
                processingTime[j] = new long[nbOperations[j]][nbMachines];
                for (int o = 0; o < nbOperations[j]; ++o) {
                    int nbMachinesOperation = input.nextInt();
                    Arrays.fill(processingTime[j][o], INFINITE);
                    for (int m = 0; m < nbMachinesOperation; ++m) {
                        int machine = input.nextInt() - 1;
                        long time = input.nextLong();
                        processingTime[j][o][machine] = time;
                    }
                    jobOperationTask[j][o] = nbTasks;
                    nbTasks++;
                }
            }
            taskSetupTimeData = new int[nbMachines][nbTasks][nbTasks];
            int maxSetup = 0;
            for (int m = 0; m < nbMachines; ++m) {
                for (int i = 0; i < nbTasks; ++i) {
                    for (int j = 0; j < nbTasks; ++j) {
                        taskSetupTimeData[m][i][j] = input.nextInt();
                        if (taskSetupTimeData[m][i][j] != INFINITE && taskSetupTimeData[m][i][j] > maxSetup)
                            maxSetup = taskSetupTimeData[m][i][j];
                    }
                }
            }

            // Trivial upper bound for the end times of the tasks
            long maxSumProcessingTimes = 0;
            taskProcessingTimeData = new long[nbTasks][];
            for (int j = 0; j < nbJobs; ++j) {
                long maxProcessingTime = 0;
                for (int o = 0; o < nbOperations[j]; ++o) {
                    int task = jobOperationTask[j][o];
                    taskProcessingTimeData[task] = new long[nbMachines];
                    for (int m = 0; m < nbMachines; ++m) {
                        taskProcessingTimeData[task][m] = processingTime[j][o][m];
                        if (processingTime[j][o][m] != INFINITE && processingTime[j][o][m] > maxProcessingTime) {
                            maxProcessingTime = processingTime[j][o][m];
                        }
                    }
                    maxSumProcessingTimes += maxProcessingTime;
                }
            }
            maxEnd = maxSumProcessingTimes + nbTasks * maxSetup;
        }
    }

    public void solve(int timeLimit) {
        // Declare the optimization model
        HxModel model = optimizer.getModel();

        // Sequence of tasks on each machine
        jobsOrder = new HxExpression[nbMachines];
        HxExpression machines = model.array();
        for (int m = 0; m < nbMachines; ++m) {
            jobsOrder[m] = model.listVar(nbTasks);
            machines.addOperand(jobsOrder[m]);
        }

        // Each task is scheduled on a machine
        model.constraint(model.partition(machines));

        // Only compatible machines can be selected for a task
        for (int i = 0; i < nbTasks; ++i) {
            for (int m = 0; m < nbMachines; ++m) {
                if (taskProcessingTimeData[i][m] == INFINITE) {
                    model.constraint(model.not(model.contains(jobsOrder[m], i)));
                }
            }
        }

        // For each task, the selected machine
        taskMachine = new HxExpression[nbTasks];
        for (int i = 0; i < nbTasks; ++i) {
            taskMachine[i] = model.find(machines, i);
        }

        HxExpression taskProcessingTime = model.array(taskProcessingTimeData);

        tasks = new HxExpression[nbTasks];
        HxExpression[] duration = new HxExpression[nbTasks];
        for (int i = 0; i < nbTasks; ++i) {
            // Interval decisions: time range of each task
            tasks[i] = model.intervalVar(0, maxEnd);

            // The task duration depends on the selected machine
            HxExpression iExpr = model.createConstant(i);
            duration[i] = model.at(taskProcessingTime, iExpr, taskMachine[i]);
            model.constraint(model.eq(model.length(tasks[i]), duration[i]));
        }
        HxExpression taskArray = model.array(tasks);

        // Precedence constraints between the operations of a job
        for (int j = 0; j < nbJobs; ++j) {
            for (int o = 0; o < nbOperations[j] - 1; ++o) {
                int i1 = jobOperationTask[j][o];
                int i2 = jobOperationTask[j][o + 1];
                model.constraint(model.lt(tasks[i1], tasks[i2]));
            }
        }

        HxExpression taskSetupTime = model.array(taskSetupTimeData);

        // Disjunctive resource constraints between the tasks on a machine
        for (int m = 0; m < nbMachines; ++m) {
            HxExpression sequence = jobsOrder[m];
            HxExpression mExpr = model.createConstant(m);
            HxExpression sequenceLambda = model
                    .lambdaFunction(
                            i -> model.geq(model.start(model.at(taskArray, model.at(sequence, model.sum(i, 1)))),
                                    model.sum(model.end(model.at(taskArray, model.at(sequence, i))),
                                            model.at(taskSetupTime, mExpr,
                                                    model.at(sequence, i), model.at(sequence, model.sum(i, 1))))));
            model.constraint(model.and(model.range(0, model.sub(model.count(sequence), 1)), sequenceLambda));
        }

        // Minimize the makespan: end of the last task
        makespan = model.max();
        for (int i = 0; i < nbTasks; ++i) {
            makespan.addOperand(model.end(tasks[i]));
        }
        model.minimize(makespan);

        model.close();

        // Parameterize the optimizer
        optimizer.getParam().setTimeLimit(timeLimit);

        optimizer.solve();
    }

    /*
     * Write the solution in a file with the following format:
     * - for each operation of each job, the selected machine, the start and end
     * dates
     */
    public void writeSolution(String fileName) throws IOException {
        try (PrintWriter output = new PrintWriter(fileName)) {
            System.out.println("Solution written in file " + fileName);

            for (int j = 1; j <= nbJobs; ++j) {
                for (int o = 1; o <= nbOperations[j - 1]; ++o) {
                    int taskIndex = jobOperationTask[j - 1][o - 1];
                    output.write(j + "\t" + o + "\t" + taskMachine[taskIndex].getValue() + "\t"
                            + tasks[taskIndex].getIntervalValue().getStart()
                            + "\t" + tasks[taskIndex].getIntervalValue().getEnd() + "\n");
                }
            }
        }
    }

    public static void main(String[] args) {
        if (args.length < 1) {
            System.out.println("Usage: java FlexibleJobshopSetup instanceFile [outputFile] [timeLimit]");
            System.exit(1);
        }

        String instanceFile = args[0];
        String outputFile = args.length > 1 ? args[1] : null;
        String strTimeLimit = args.length > 2 ? args[2] : "60";

        try (HexalyOptimizer optimizer = new HexalyOptimizer()) {
            FlexibleJobshopSetup model = new FlexibleJobshopSetup(optimizer);
            model.readInstance(instanceFile);
            model.solve(Integer.parseInt(strTimeLimit));
            if (outputFile != null) {
                model.writeSolution(outputFile);
            }
        } catch (Exception ex) {
            System.err.println(ex);
            ex.printStackTrace();
            System.exit(1);
        }
    }
}
