#include "optimizer/hexalyoptimizer.h"
#include <algorithm>
#include <fstream>
#include <iostream>
#include <limits>
#include <numeric>
#include <vector>

using namespace hexaly;

class FlexibleJobshopSetup {
  private:
    // Number of jobs
    int nbJobs;
    // Number of machines
    int nbMachines;
    // Number of tasks
    int nbTasks;
    // Processing time for each task, for each machine
    std::vector<std::vector<int>> taskProcessingTimeData;
    // Setup time between every two consecutive tasks, for each machine
    std::vector<std::vector<std::vector<int>>> taskSetupTimeData;
    // For each job, for each operation, the corresponding task id
    std::vector<std::vector<int>> jobOperationTask;
    // Number of operations for each job
    std::vector<int> nbOperations;
    // Trivial upper bound for the end times of the tasks
    int maxEnd;
    // Constant for incompatible machines
    const int INFINITE = 1000000;

    // Hexaly Optimizer
    HexalyOptimizer optimizer;
    // Decision variables: time range of each task
    std::vector<HxExpression> tasks;
    // Decision variables: sequence of tasks on each machine
    std::vector<HxExpression> jobsOrder;
    // For each task, the selected machine
    std::vector<HxExpression> taskMachine;
    // Objective = minimize the makespan: end of the last task
    HxExpression makespan;

  public:
    FlexibleJobshopSetup() : optimizer() {}

    void readInstance(const std::string& fileName) {
        std::ifstream infile;
        infile.exceptions(std::ifstream::failbit | std::ifstream::badbit);
        infile.open(fileName.c_str());

        infile >> nbJobs;
        infile >> nbMachines;
        infile.ignore(std::numeric_limits<std::streamsize>::max(), '\n'); // skip last number

        nbTasks = 0;
        std::vector<std::vector<std::vector<int>>> processingTime = std::vector<std::vector<std::vector<int>>>(nbJobs);
        jobOperationTask.resize(nbJobs);
        nbOperations.resize(nbJobs);
        for (unsigned int j = 0; j < nbJobs; ++j) {
            infile >> nbOperations[j];
            jobOperationTask[j].resize(nbOperations[j]);
            processingTime[j].resize(nbOperations[j]);
            for (unsigned int o = 0; o < nbOperations[j]; ++o) {
                int nbMachinesOperation;
                infile >> nbMachinesOperation;
                taskProcessingTimeData.push_back(std::vector<int>(nbMachines, INFINITE));
                processingTime[j][o].resize(nbMachines, INFINITE);
                for (int m = 0; m < nbMachinesOperation; ++m) {
                    int machine;
                    int time;
                    infile >> machine;
                    infile >> time;
                    processingTime[j][o][machine - 1] = time;
                    taskProcessingTimeData[nbTasks][machine - 1] = time;
                }
                jobOperationTask[j][o] = nbTasks;
                nbTasks += 1;
            }
        }

        taskSetupTimeData = std::vector<std::vector<std::vector<int>>>(
            nbMachines, std::vector<std::vector<int>>(nbTasks, std::vector<int>(nbTasks)));
        int maxSetup = 0;
        for (unsigned int m = 0; m < nbMachines; m++) {
            for (unsigned int i = 0; i < nbTasks; i++) {
                for (unsigned int j = 0; j < nbTasks; j++) {
                    infile >> taskSetupTimeData[m][i][j];
                    if (taskSetupTimeData[m][i][j] != INFINITE && taskSetupTimeData[m][i][j] > maxSetup)
                        maxSetup = taskSetupTimeData[m][i][j];
                }
            }
        }

        infile.close();

        // Trivial upper bound for the end times of the tasks
        int maxSumProcessingTimes = 0;
        for (unsigned int j = 0; j < nbJobs; ++j) {
            for (unsigned int o = 0; o < nbOperations[j]; ++o) {
                int maxProcessingTime = 0;
                for (unsigned int m = 0; m < nbMachines; ++m) {
                    if (processingTime[j][o][m] != INFINITE && processingTime[j][o][m] > maxProcessingTime)
                        maxProcessingTime = processingTime[j][o][m];
                }
                maxSumProcessingTimes += maxProcessingTime;
            }
        }
        maxEnd = maxSumProcessingTimes + nbTasks * maxSetup;
    }

    void solve(int timeLimit) {
        // Declare the optimization model
        HxModel model = optimizer.getModel();

        // Sequence of tasks on each machine
        jobsOrder.resize(nbMachines);
        HxExpression machines = model.array();
        for (unsigned int m = 0; m < nbMachines; ++m) {
            jobsOrder[m] = model.listVar(nbTasks);
            machines.addOperand(jobsOrder[m]);
        }

        // Each task is scheduled on a machine
        model.constraint(model.partition(machines));

        // Only compatible machines can be selected for a task
        for (int i = 0; i < nbTasks; ++i) {
            for (unsigned int m = 0; m < nbMachines; ++m) {
                if (taskProcessingTimeData[i][m] == INFINITE) {
                    model.constraint(!model.contains(jobsOrder[m], i));
                }
            }
        }

        taskMachine.resize(nbTasks);
        HxExpression taskProcessingTime = model.array();
        for (int i = 0; i < nbTasks; ++i) {
            // For each task, the selected machine
            taskMachine[i] = model.find(machines, i);
            taskProcessingTime.addOperand(
                model.array(taskProcessingTimeData[i].begin(), taskProcessingTimeData[i].end()));
        }

        HxExpression taskSetupTime = model.array();
        for (int m = 0; m < nbMachines; ++m) {
            HxExpression setupTimeMachine = model.array();
            for (int i = 0; i < nbTasks; ++i) {
                setupTimeMachine.addOperand(
                    model.array(taskSetupTimeData[m][i].begin(), taskSetupTimeData[m][i].end()));
            }
            taskSetupTime.addOperand(setupTimeMachine);
        }

        tasks.resize(nbTasks);
        std::vector<HxExpression> duration(nbTasks);
        for (int i = 0; i < nbTasks; ++i) {
            // Interval decisions: time range of each task
            tasks[i] = model.intervalVar(0, maxEnd);

            // The task duration depends on the selected machine
            duration[i] = model.at(taskProcessingTime, i, taskMachine[i]);
            model.constraint(model.length(tasks[i]) == duration[i]);
        }
        HxExpression taskArray = model.array(tasks.begin(), tasks.end());

        // Precedence constraints between the operations of a job
        for (unsigned int j = 0; j < nbJobs; ++j) {
            for (unsigned int o = 0; o < nbOperations[j] - 1; ++o) {
                int i1 = jobOperationTask[j][o];
                int i2 = jobOperationTask[j][o + 1];
                model.constraint(taskArray[i1] < taskArray[i2]);
            }
        }

        // Disjunctive resource constraints between the tasks on a machine
        for (int m = 0; m < nbMachines; ++m) {
            HxExpression sequence = jobsOrder[m];
            HxExpression sequenceLambda = model.createLambdaFunction([&](HxExpression i) {
                return model.start(taskArray[sequence[i + 1]]) >=
                       model.end(taskArray[sequence[i]]) + model.at(taskSetupTime, m, sequence[i], sequence[i + 1]);
            });
            model.constraint(model.and_(model.range(0, model.count(sequence) - 1), sequenceLambda));
        }

        // Minimize the makespan: end of the last task
        makespan = model.max();
        for (int i = 0; i < nbTasks; ++i) {
            makespan.addOperand(model.end(tasks[i]));
        }
        model.minimize(makespan);

        model.close();

        // Parameterize the optimizer
        optimizer.getParam().setTimeLimit(timeLimit);

        optimizer.solve();
    }

    /* Write the solution in a file with the following format:
     *  - for each operation of each job, the selected machine, the start and end dates */
    void writeSolution(const std::string& fileName) {
        std::ofstream outfile(fileName.c_str());
        if (!outfile.is_open()) {
            std::cerr << "File " << fileName << " cannot be opened." << std::endl;
            exit(1);
        }
        std::cout << "Solution written in file " << fileName << std::endl;

        for (unsigned int j = 0; j < nbJobs; ++j) {
            for (unsigned int o = 0; o < nbOperations[j]; ++o) {
                int taskIndex = jobOperationTask[j][o];
                outfile << j + 1 << "\t" << o + 1 << "\t" << taskMachine[taskIndex].getValue() + 1 << "\t"
                        << tasks[taskIndex].getIntervalValue().getStart() << "\t"
                        << tasks[taskIndex].getIntervalValue().getEnd() << std::endl;
            }
        }
        outfile.close();
    }
};

int main(int argc, char** argv) {
    if (argc < 2) {
        std::cout << "Usage: flexible_jobshop_setup instanceFile [outputFile] [timeLimit]" << std::endl;
        exit(1);
    }

    const char* instanceFile = argv[1];
    const char* outputFile = argc > 2 ? argv[2] : NULL;
    const char* strTimeLimit = argc > 3 ? argv[3] : "60";

    FlexibleJobshopSetup model;
    try {
        model.readInstance(instanceFile);
        const int timeLimit = atoi(strTimeLimit);
        model.solve(timeLimit);
        if (outputFile != NULL)
            model.writeSolution(outputFile);
        return 0;
    } catch (const std::exception& e) {
        std::cerr << "An error occurred: " << e.what() << std::endl;
        return 1;
    }
}
