using System;
using System.IO;
using Hexaly.Optimizer;

public class Flowshop : IDisposable
{
    // Number of jobs
    int nbJobs;

    // Number of machines
    int nbMachines;

    // Processing time
    long[][] processingTimeData;

    // Hexaly Optimizer
    HexalyOptimizer optimizer;

    // Decision variable
    HxExpression jobs;

    // Objective
    HxExpression makespan;

    public Flowshop()
    {
        optimizer = new HexalyOptimizer();
    }

    /* Read instance data */
    void ReadInstance(string fileName)
    {
        using (StreamReader input = new StreamReader(fileName))
        {
            string[] firstLineSplit = input
                .ReadLine()
                .Split((char[])null, StringSplitOptions.RemoveEmptyEntries);
            nbJobs = int.Parse(firstLineSplit[0]);
            nbMachines = int.Parse(firstLineSplit[1]);

            string[] matrixText = input
                .ReadToEnd()
                .Split((char[])null, StringSplitOptions.RemoveEmptyEntries);
            processingTimeData = new long[nbMachines][];
            for (int m = 0; m < nbMachines; ++m)
            {
                processingTimeData[m] = new long[nbJobs];
                for (int j = 0; j < nbJobs; ++j)
                    processingTimeData[m][j] = long.Parse(matrixText[m * nbJobs + j]);
            }
        }
    }

    public void Dispose()
    {
        if (optimizer != null)
            optimizer.Dispose();
    }

    void Solve(int limit)
    {
        // Declare the optimization model
        HxModel model = optimizer.GetModel();

        // Permutation of jobs
        jobs = model.List(nbJobs);

        // All jobs have to be assigned
        model.Constraint(model.Count(jobs) == nbJobs);

        // For each machine create proccessingTime[m] as an array to be able to access it
        // with an 'at' operator
        HxExpression[] processingTime = new HxExpression[nbMachines];
        for (int m = 0; m < nbMachines; ++m)
            processingTime[m] = model.Array(processingTimeData[m]);

        // On machine 0, the jth job ends on the time it took to be processed after
        // the end of the previous job
        HxExpression[] jobEnd = new HxExpression[nbJobs];
        HxExpression firstEndLambda = model.LambdaFunction(
            (i, prev) => prev + processingTime[0][jobs[i]]
        );
        jobEnd[0] = model.Array(model.Range(0, nbJobs), firstEndLambda, 0);

        // The jth job on machine m starts when it has been processed by machine n-1
        // AND when job j-1 has been processed on machine m. It ends after it has been processed.
        for (int m = 1; m < nbMachines; ++m)
        {
            HxExpression endLambda = model.LambdaFunction(
                (i, prev) => model.Max(prev, jobEnd[m - 1][i]) + processingTime[m][jobs[i]]
            );
            jobEnd[m] = model.Array(model.Range(0, nbJobs), endLambda, 0);
        }

        // Minimize the makespan: end of the last job on the last machine
        makespan = jobEnd[nbMachines - 1][nbJobs - 1];
        model.Minimize(makespan);

        model.Close();

        // Parametrize the optimizer
        optimizer.GetParam().SetTimeLimit(limit);

        optimizer.Solve();
    }

    /* Write the solution in a file */
    void WriteSolution(string fileName)
    {
        using (StreamWriter output = new StreamWriter(fileName))
        {
            output.WriteLine(makespan.GetValue());
            HxCollection jobsCollection = jobs.GetCollectionValue();
            for (int j = 0; j < nbJobs; ++j)
                output.Write(jobsCollection[j] + " ");
            output.WriteLine();
        }
    }

    public static void Main(string[] args)
    {
        if (args.Length < 1)
        {
            Console.WriteLine("Usage: Flowshop inputFile [solFile] [timeLimit]");
            Environment.Exit(1);
        }
        string instanceFile = args[0];
        string outputFile = args.Length > 1 ? args[1] : null;
        string strTimeLimit = args.Length > 2 ? args[2] : "5";

        using (Flowshop model = new Flowshop())
        {
            model.ReadInstance(instanceFile);
            model.Solve(int.Parse(strTimeLimit));
            if (outputFile != null)
                model.WriteSolution(outputFile);
        }
    }
}
