#include "optimizer/hexalyoptimizer.h"
#include <fstream>
#include <iostream>
#include <vector>

using namespace hexaly;
using namespace std;

class Flowshop {
public:
    // Number of jobs
    int nbJobs;

    // Number of machines
    int nbMachines;

    // Processing time
    vector<vector<int>> processingTimeData;

    // Hexaly Optimizer
    HexalyOptimizer optimizer;

    // Decision variable
    HxExpression jobs;

    // Objective
    HxExpression makespan;

    /* Read instance data */
    void readInstance(const string& fileName) {
        ifstream infile;
        infile.exceptions(ifstream::failbit | ifstream::badbit);
        infile.open(fileName.c_str());

        long tmp;
        infile >> nbJobs;
        infile >> nbMachines;
        infile >> tmp;
        infile >> tmp;
        infile >> tmp;

        processingTimeData.resize(nbMachines);
        for (int m = 0; m < nbMachines; ++m) {
            processingTimeData[m].resize(nbJobs);
            for (int j = 0; j < nbJobs; ++j) {
                infile >> processingTimeData[m][j];
            }
        }
    }

    void solve(int limit) {
        // Declare the optimization model
        HxModel model = optimizer.getModel();

        // Permutation of jobs
        jobs = model.listVar(nbJobs);

        // All jobs have to be assigned
        model.constraint(model.count(jobs) == nbJobs);

        // For each machine create proccessingTime[m] as an array to be able to access it
        // with an 'at' operator
        vector<HxExpression> processingTime(nbMachines);
        for (int m = 0; m < nbMachines; ++m) {
            processingTime[m] = model.array(processingTimeData[m].begin(), processingTimeData[m].end());
        }

        // On machine 0, the jth job ends on the time it took to be processed after
        // the end of the previous job
        vector<HxExpression> jobEnd(nbMachines);
        HxExpression firstEndLambda = model.createLambdaFunction(
            [&](HxExpression i, HxExpression prev) { return prev + processingTime[0][jobs[i]]; });
        jobEnd[0] = model.array(model.range(0, nbJobs), firstEndLambda, 0);

        // The jth job on machine m starts when it has been processed by machine n-1
        // AND when job j-1 has been processed on machine m. It ends after it has been processed.
        for (int m = 1; m < nbMachines; ++m) {
            int mL = m;
            HxExpression endLambda = model.createLambdaFunction([&](HxExpression i, HxExpression prev) {
                return model.max(prev, jobEnd[mL - 1][i]) + processingTime[mL][jobs[i]];
            });
            jobEnd[m] = model.array(model.range(0, nbJobs), endLambda, 0);
        }

        // Minimize the makespan: end of the last job on the last machine
        makespan = jobEnd[nbMachines - 1][nbJobs - 1];
        model.minimize(makespan);
        model.close();

        // Parametrize the optimizer
        optimizer.getParam().setTimeLimit(limit);

        optimizer.solve();
    }

    /* Write the solution in a file */
    void writeSolution(const string& fileName) {
        ofstream outfile;
        outfile.exceptions(ofstream::failbit | ofstream::badbit);
        outfile.open(fileName.c_str());

        outfile << makespan.getValue() << endl;
        HxCollection jobsCollection = jobs.getCollectionValue();
        for (int j = 0; j < nbJobs; ++j) {
            outfile << jobsCollection[j] << " ";
        }
        outfile << endl;
    }
};

int main(int argc, char** argv) {
    if (argc < 2) {
        cerr << "Usage: flowshop inputFile [outputFile] [timeLimit]" << endl;
        return 1;
    }

    const char* instanceFile = argv[1];
    const char* solFile = argc > 2 ? argv[2] : NULL;
    const char* strTimeLimit = argc > 3 ? argv[3] : "5";

    try {
        Flowshop model;
        model.readInstance(instanceFile);
        model.solve(atoi(strTimeLimit));
        if (solFile != NULL)
            model.writeSolution(solFile);
        return 0;
    } catch (const exception& e) {
        cerr << "An error occurred: " << e.what() << endl;
        return 1;
    }
}
