using System;
using System.IO;
using Hexaly.Optimizer;

public class Frcpsps : IDisposable
{
    // Number of tasks
    private int nbTasks;
    // Number of resources
    private int nbResources;
    // Number of states
    private int nbStates;
    // Maximum capacity of each resource
    private int[] capacity;
    // Delay after change of state
    private int[][] delayState;
    // Duration of each task
    private int[] duration;
    // state of each task
    private int[] state;

    // Trivial upper bound for the end times of the tasks
    private int horizon = 0;
    // Hexaly Optimizer
    private HexalyOptimizer optimizer;
    // Decision variables: time range of each task
    private HxExpression[] tasks;
    // Decision variables: set of tasks done by each resource
    private HxExpression[] resourceTasks;
    // For each task, the selected resource
    private HxExpression[] taskResource;    

    // Objective = minimize the makespan: end of the last task of the last job
    private HxExpression makespan;

    public Frcpsps(string fileName)
    {
        optimizer = new HexalyOptimizer();
    }

    private static string[] ReadNextLine(StreamReader input) {
        return input.ReadLine().Split(new[] { ' ' }, StringSplitOptions.RemoveEmptyEntries);
    }


    private void ReadInstance(string fileName)
    {
        using (StreamReader input = new StreamReader(fileName))
        {
            string[] splitted = ReadNextLine(input);
            if (splitted.Length == 0)
                splitted = ReadNextLine(input);
            nbTasks = int.Parse(splitted[0]);
            nbResources = int.Parse(splitted[1]);
            nbStates = int.Parse(splitted[2]);


            capacity = new int[nbResources];
            splitted = ReadNextLine(input);
            for (int r = 0; r < nbResources; ++r)
                capacity[r] = int.Parse(splitted[r]);

            delayState = new int[nbStates][];
            for (int i = 0; i < nbStates; ++i)
            {
                splitted = ReadNextLine(input);
                delayState[i] = new int[nbStates];
                for (int j = 0; j < nbStates; ++j)
                    delayState[i][j] = int.Parse(splitted[j]);
            }

            duration = new int[nbTasks];
            state = new int[nbTasks];

            for (int i = 0; i < nbTasks; ++i)
            {
                splitted = ReadNextLine(input);
                if (splitted.Length == 0)
                    splitted = ReadNextLine(input);
                duration[i] = int.Parse(splitted[0]);
                state[i] = int.Parse(splitted[1]);
                horizon += duration[i];
            }
        }
    }

    public void Dispose()
    {
        optimizer.Dispose();
    }

    public void Solve(int timeLimit)
    {
        // Declare the optimization model
        HxModel model = optimizer.GetModel();

        // Interval decisions: time range of each task
        tasks = new HxExpression[nbTasks];
        for (int i = 0; i < nbTasks; ++i)
        {
            tasks[i] = model.Interval(0, horizon);

        // Task duration constraints
            model.Constraint(model.Length(tasks[i]) == duration[i]);
        }

        // Set of tasks done by each resource
        resourceTasks = new HxExpression[nbResources];
        for (int r = 0; r < nbResources; ++r)
            resourceTasks[r] = model.Set(nbTasks);

        // Creates Hexaly arrays to allow access through "at" operator
        HxExpression resources = model.Array(resourceTasks);
        HxExpression delayArray = model.Array(delayState);
        HxExpression tasksArray = model.Array(tasks);
        HxExpression stateArray = model.Array(state);

        // All tasks must be sheduled on a resource
        model.Constraint(model.Partition(resources));

        // For each task, the selected resource
        taskResource = new HxExpression[nbTasks];
        for (int i = 0; i < nbTasks; ++i)
        {
            taskResource[i] = model.Find(resources, i);
        }

        // Makespan: end of the last task
        makespan = model.Max();
        for (int i = 0; i < nbTasks; ++i)
            makespan.AddOperand(model.End(tasks[i]));

        // Resource constraints
        for (int r = 0; r < nbResources; ++r)
        {
            HxExpression capacityRespected = model.LambdaFunction(t =>
                {
                    HxExpression total = model.LambdaFunction(i =>
                    {
                        HxExpression rExpr = model.CreateConstant(r);
                        return model.Contains(tasksArray[i], t);
                    });
                    HxExpression totalTasks = model.Sum(resourceTasks[r], total);
                    return totalTasks <= capacity[r];
                });
            model.Constraint(model.And(model.Range(0, makespan), capacityRespected));
        }

        // State incompatibility constraints
        for (int r = 0; r < nbResources; ++r)
        {
            HxExpression stateRespected = model.LambdaFunction(j =>
                {
                    HxExpression total = model.LambdaFunction(i =>
                    {
                        HxExpression rExpr = model.CreateConstant(r);
                        return model.Or(stateArray[i]==stateArray[j],
                                        (model.End(tasksArray[i]) + delayArray[stateArray[i]][stateArray[j]]) <= model.Start(tasksArray[j]),
                                        (model.End(tasksArray[j]) + delayArray[stateArray[j]][stateArray[i]]) <= model.Start(tasksArray[i]));
                    });
                    HxExpression TotalExpr = model.And(resourceTasks[r], total);
                    return TotalExpr;                                                    
                });
            model.Constraint(model.And(resourceTasks[r], stateRespected));
        }

        // Minimize the makespan
        model.Minimize(makespan);

        model.Close();

        // Parameterize the optimizer
        optimizer.GetParam().SetTimeLimit(timeLimit);

        optimizer.Solve();
    }

    /* Write the solution in a file with the following format:
    *  - total makespan, number of Tasks
    *  - for each task, the task id, the start, the end times and the ressource of the task*/
    public void WriteSolution(string fileName)
    {
        using (StreamWriter output = new StreamWriter(fileName))
        {
            Console.WriteLine("Solution written in file " + fileName);
            output.WriteLine(makespan.GetValue()+ " " + nbTasks);
            for (int i = 0; i < nbTasks; ++i)
            {
                output.Write((i + 1) + " " + tasks[i].GetIntervalValue().Start() + " " + tasks[i].GetIntervalValue().End() + " " + taskResource[i].GetValue());
                output.WriteLine();
            }
        }
    }

    public static void Main(string[] args)
    {
        if (args.Length < 1)
        {
            Console.WriteLine("Usage: Frcpsps instanceFile [outputFile] [timeLimit]");
            System.Environment.Exit(1);
        }

        string instanceFile = args[0];
        string outputFile = args.Length > 1 ? args[1] : null;
        string strTimeLimit = args.Length > 2 ? args[2] : "60";

        using (Frcpsps model = new Frcpsps(instanceFile))
        {
            model.ReadInstance(instanceFile);
            model.Solve(int.Parse(strTimeLimit));
            if (outputFile != null)
                model.WriteSolution(outputFile);
        }
    }
}
