import java.util.*;
import java.io.*;
import com.hexaly.optimizer.*;

public class Frcpsps {    
    // Number of tasks
    private int nbTasks;
    // Number of resources
    private int nbResources;
    // Number of states
    private int nbStates;
    // Maximum capacity of each resource
    private int[] capacity;
    // Delay after change of state
    private int[][] delayState;
    // Duration of each task
    private int[] duration;
    // state of each task
    private int[] state;
    // Trivial upper bound for the end times of the tasks
    private int horizon = 0;

    // Hexaly Optimizer
    final HexalyOptimizer optimizer;
    // Decision variables: time range of each task
    private HxExpression[] tasks;
    // Decision variables: set of tasks done by each resource
    private HxExpression[] resourceTasks;
    // For each task, the selected resource
    private HxExpression[] taskResource;
    // Objective = minimize the makespan: end of the last task of the last job
    private HxExpression makespan;

    public Frcpsps(HexalyOptimizer optimizer, String fileName) throws IOException {
        this.optimizer = optimizer;
    }

    private void readInstance(String fileName) throws IOException {
        try (Scanner input = new Scanner(new File(fileName))) {
            input.useLocale(Locale.ROOT);
            nbTasks = input.nextInt();
            nbResources = input.nextInt();
            nbStates = input.nextInt();


            capacity = new int[nbResources];
            for (int r = 0; r < nbResources; ++r) {
                capacity[r] = input.nextInt();
            }

            delayState = new int[nbStates][nbStates];
            for (int i = 0; i < nbStates; ++i)  {
                for (int j = 0; j < nbStates; ++j)  {
                delayState[i][j] = input.nextInt();
                }
            }

            duration = new int[nbTasks];
            state = new int[nbTasks];
            
            for (int i = 0; i < nbTasks; ++i) {
                duration[i] = input.nextInt();
                state[i] = input.nextInt();
                horizon += duration[i];
            }
        }
    }

    public void solve(int timeLimit) {
        // Declare the optimization model
        HxModel model = optimizer.getModel();

        // Interval decisions: time range of each task
        tasks = new HxExpression[nbTasks];
        for (int i = 0; i < nbTasks; ++i) {
            tasks[i] = model.intervalVar(0, horizon);

        // Task duration constraints
            model.constraint(model.eq(model.length(tasks[i]), duration[i]));
        }

        // Set of tasks done by each resource
        resourceTasks = new HxExpression[nbResources];

        for (int r = 0; r < nbResources; ++r) {
            resourceTasks[r] = model.setVar(nbTasks);
        }

        // Creates Hexaly arrays to allow access through "at" operator
        HxExpression resources = model.array(resourceTasks);
        HxExpression delayArray = model.array(delayState);
        HxExpression tasksArray = model.array(tasks);
        HxExpression stateArray = model.array(state);

        // For each task, the selected resource
        taskResource = new HxExpression[nbTasks];
        for (int i = 0; i < nbTasks; ++i) {
            taskResource[i] = model.find(resources, i);
        }
        
        // All tasks must be sheduled on a resource
        model.constraint(model.partition(resources));        

        // Makespan: end of the last task
        makespan = model.max();
        for (int i = 0; i < nbTasks; ++i) {
            makespan.addOperand(model.end(tasks[i]));
        }

        // Resource constraints
        for (int r = 0; r < nbResources; ++r) {
            final int rL = r;
            HxExpression capacityRespected = model.lambdaFunction(t -> {
                HxExpression total = model.lambdaFunction(i -> {
                    HxExpression rExpr = model.createConstant(rL);
                    return model.contains(model.at(tasksArray, i), t);
                });
                HxExpression totalTasks = model.sum(resourceTasks[rL], total);
                return model.leq(totalTasks, capacity[rL]);
            });
            model.constraint(model.and(model.range(0, makespan), capacityRespected));
        }

        // State incompatibility constraints
        for (int r = 0; r < nbResources; ++r) {
            final int rL = r;
            HxExpression stateRespected = model.lambdaFunction(j -> {
                    HxExpression total = model.lambdaFunction(i ->
                    {
                        HxExpression rExpr = model.createConstant(rL);
                        return model.or(
                            model.eq(model.at(stateArray, i), model.at(stateArray, j)), 
                            model.leq(
                                model.sum(model.end(model.at(tasksArray, j)), model.at(delayArray, model.at(stateArray, j), model.at(stateArray, i))), 
                                model.start(model.at(tasksArray, i))),
                            model.leq(
                                model.sum(model.end(model.at(tasksArray, i)), model.at(delayArray, model.at(stateArray, i), model.at(stateArray, j))), 
                                model.start(model.at(tasksArray, j)))
                            );
                                        
                    });
                    HxExpression TotalExpr = model.and(resourceTasks[rL], total);
                    return TotalExpr;
                                                    
                });
            model.constraint(model.and(resourceTasks[rL], stateRespected));
        }

        // Minimize the makespan
        model.minimize(makespan);

        model.close();

        // Parameterize the optimizer
        optimizer.getParam().setTimeLimit(timeLimit);

        optimizer.solve();
    }

    /* Write the solution in a file with the following format:
    *  - total makespan, number of Tasks
    *  - for each task, the task id, the start, the end times and the ressource of the task*/
    public void writeSolution(String fileName) throws IOException {
        try (PrintWriter output = new PrintWriter(fileName)) {
            System.out.println("Solution written in file " + fileName);

            output.println(makespan.getValue()+ " " + nbTasks);

            for (int i = 0; i < nbTasks; ++i) {
                output.println((i + 1) + " " + tasks[i].getIntervalValue().getStart() + " "
                        + tasks[i].getIntervalValue().getEnd() + " "
                        + taskResource[i].getValue()   
                        );
            }
        }
    }

    public static void main(String[] args) {
        if (args.length < 1) {
            System.out.println("Usage: java project_schedulingg instanceFile [outputFile] [timeLimit]");
            System.exit(1);
        }

        String instanceFile = args[0];
        String outputFile = args.length > 1 ? args[1] : null;
        String strTimeLimit = args.length > 2 ? args[2] : "60";
        try (HexalyOptimizer optimizer = new HexalyOptimizer()) {
            Frcpsps model = new Frcpsps(optimizer, instanceFile);
            model.readInstance(instanceFile);
            model.solve(Integer.parseInt(strTimeLimit));
            if (outputFile != null) {
                model.writeSolution(outputFile);
            }
        } catch (Exception ex) {
            System.err.println(ex);
            ex.printStackTrace();
            System.exit(1);
        }
    }
}
