#include "optimizer/hexalyoptimizer.h"
#include <algorithm>
#include <fstream>
#include <iostream>
#include <limits>
#include <numeric>
#include <vector>


using namespace hexaly;

class Frcpsps {
private:
    // Number of tasks
    int nbTasks;
    // Number of resources
    int nbResources;
    // Number of states
    int nbStates;
    // Maximum capacity of each resource
    std::vector<int> capacity;
    // Delay after change of state
    std::vector<std::vector<int>> delayState;
    // Duration of each task
    std::vector<int> duration;
    // state of each task
    std::vector<int> state;
    // Trivial upper bound for the end times of the tasks
    int horizon = 0;

    // Hexaly Optimizer
    HexalyOptimizer optimizer;
    
    // Decision variables: time range of each task
    std::vector<HxExpression> tasks;

    // Decision variables: set of tasks done by each resource
    std::vector<HxExpression> resourceTasks;

    // For each task, the selected resource
    std::vector<HxExpression> taskResource;
    
    // Objective = minimize the makespan: end of the last task
    HxExpression makespan;


public:
    Frcpsps(const std::string& fileName) : optimizer() {}

    void readInstance(const std::string& fileName) {
        std::ifstream infile;
        infile.exceptions(std::ifstream::failbit | std::ifstream::badbit);
        infile.open(fileName.c_str());

        infile >> nbTasks >> nbResources >> nbStates;

        capacity.resize(nbResources);
        for (int r = 0; r < nbResources; ++r) {
            infile >> capacity[r];
        }

        delayState.resize(nbStates);
        for (int s = 0; s < nbStates; ++s) {
            delayState[s].resize(nbStates);
        }

        for (int i = 0; i < nbStates; ++i) {
            for (int j = 0; j < nbStates; ++j) {
                infile >> delayState[i][j];
            }
        }

        duration.resize(nbTasks);
        state.resize(nbTasks);
        for (int i = 0; i < nbTasks; ++i) {
            infile >> duration[i] >> state[i];
        }

        // Trivial upper bound for the end times of the tasks
        horizon = std::accumulate(duration.begin(), duration.end(), 0);

        infile.close();
    }

    void solve(int TimeLimit) {
        // Declare the optimization model
        HxModel model = optimizer.getModel();

        // Interval decisions: time range of each task
        tasks.resize(nbTasks);
        for (int i = 0; i < nbTasks; ++i) {
            tasks[i] = model.intervalVar(0, horizon);
            // Task duration constraints
            model.constraint(model.length(tasks[i]) == duration[i]);
        }

        // Set of tasks done by each resource
        resourceTasks.resize(nbResources);
        HxExpression resources = model.array();
        for (int r = 0; r < nbResources; ++r) {
            resourceTasks[r] = model.setVar(nbTasks);
            resources.addOperand(resourceTasks[r]);
        }
        
        // All tasks must be sheduled on a resource
        model.constraint(model.partition(resources));

        // Makespan: end of the last task
        makespan = model.max();
        for (int i = 0; i < nbTasks; ++i) {
            makespan.addOperand(model.end(tasks[i]));
        }

        // For each task, the selected resource
        taskResource.resize(nbTasks);
        for (int i = 0; i < nbTasks; ++i) {
            taskResource[i] = model.find(resources, i);
        }

        // Creates Hexaly arrays to allow access through "at" operator
        HxExpression tasksArray = model.array();
        for (int i = 0; i < nbTasks; ++i) {
            tasksArray.addOperand(tasks[i]);
        }
        HxExpression stateArray = model.array();
        for (int i = 0; i < nbTasks; ++i) {
            stateArray.addOperand(state[i]);
        }
        HxExpression delayArray = model.array();
        for (int s = 0; s < nbStates; ++s) {
            delayArray.addOperand(model.array(delayState[s].begin(), delayState[s].end()));
        }

        // Resource constraints
        for (int r = 0; r < nbResources; ++r) {
            HxExpression capacityRespected = model.createLambdaFunction([&](HxExpression t) {
                HxExpression total = model.createLambdaFunction([&](HxExpression i) {
                    return model.contains(tasksArray[i], t);
                });
                HxExpression totalTasks = model.sum(resourceTasks[r], total);
                return model.leq(totalTasks, capacity[r]);
            });
            model.constraint(model.and_(model.range(0, makespan), capacityRespected));
        }

        // State incompatibility constraints
        for (int r = 0; r < nbResources; ++r) {
            HxExpression stateRespected = model.createLambdaFunction([&](HxExpression j) {
                HxExpression total = model.createLambdaFunction([&](HxExpression i) {
                    return model.or_(stateArray[i]==stateArray[j],
                                        model.end(tasksArray[i]) + delayArray[stateArray[i]][stateArray[j]] <= model.start(tasksArray[j]),
                                        model.end(tasksArray[j]) + delayArray[stateArray[j]][stateArray[i]] <= model.start(tasksArray[i]));
                });
                HxExpression totalstates = model.and_(resourceTasks[r], total);
                return  totalstates;
            });
            model.constraint(model.and_(resourceTasks[r], stateRespected));
        }

        // Minimize the makespan
        model.minimize(makespan);

        model.close();

        // Parameterize the optimizer
        optimizer.getParam().setTimeLimit(TimeLimit);

        optimizer.solve();        
    }

    /* Write the solution in a file with the following format:
    *  - total makespan, number of Tasks
    *  - for each task, the task id, the start, the end times and the ressource of the task*/
    void writeSolution(const std::string& fileName) {
        std::ofstream outfile(fileName.c_str());
        if (!outfile.is_open()) {
            std::cerr << "File " << fileName << " cannot be opened." << std::endl;
            exit(1);
        }
        std::cout << "Solution written in file " << fileName << std::endl;

        outfile << makespan.getValue() << " " << nbTasks << std::endl;
        for (int i = 0; i < nbTasks; ++i) {
            outfile << i + 1 << " " << tasks[i].getIntervalValue().getStart() << " "
                    << tasks[i].getIntervalValue().getEnd() << " "
                    << taskResource[i].getValue()
                    << std::endl;
        }
        outfile.close();
    }
};

int main(int argc, char** argv) {
    if (argc < 2) {
        std::cout << "Usage: Frcpsps instanceFile [outputFile] [timeLimit]" << std::endl;
        exit(1);
    }

    const char* instanceFile = argv[1];
    const char* outputFile = argc > 2 ? argv[2] : NULL;
    const char* strTimeLimit = argc > 3 ? argv[3] : "60";

    Frcpsps model(instanceFile);
    try {
        model.readInstance(instanceFile);
        const int timeLimit = atoi(strTimeLimit);
        model.solve(timeLimit);
        if (outputFile != NULL)
            model.writeSolution(outputFile);
        return 0;
    } catch (const std::exception& e) {
        std::cerr << "An error occurred: " << e.what() << std::endl;
        return 1;
    }
}
