import hexaly.optimizer
import sys

def read_instance(filename):
    with open(filename) as f:
        lines = f.readlines()

    first_line = lines[0].split()

    # Number of tasks
    nb_tasks = int(first_line[0])

    # Number of resources
    nb_resources = int(first_line[1])

    # Number of states
    nb_states = int(first_line[2])

    # Maximum capacity of each resource
    capacity = [int(lines[1].split()[r]) for r in range(nb_resources)]

    #Delay after change of state
    delay_state = [[] for i in range(nb_states)]
    for i in range(nb_states):
        delay_state[i] = [int(lines[2+i].split()[j]) for j in range(nb_states)]

    # Duration of each task
    duration = [0 for i in range(nb_tasks)]

    # State of each task
    state = [0 for i in range(nb_tasks)]

    for i in range(nb_tasks):
        task_line = lines[i + 2 + nb_states].split()
        duration[i] = int(task_line[0])
        state[i] = int(task_line[1])
    
    # Trivial upper bound for the end times of the tasks
    horizon = sum(duration[i] for i in range(nb_tasks))

    return (nb_tasks, nb_resources, nb_states, capacity, delay_state, duration, state , horizon)


def main(instance_file, output_file, time_limit):
    nb_tasks, nb_resources, nb_states, capacity, delay_state, duration, state , horizon = read_instance(instance_file)

    with hexaly.optimizer.HexalyOptimizer() as optimizer:
        # Declare the optimization model
        model = optimizer.model

        # Interval decisions: time range of each task
        tasks = [model.interval(0, horizon) for i in range(nb_tasks)]

        # Task duration constraints
        for i in range(nb_tasks):
            model.constraint(model.length(tasks[i]) == duration[i])

        # Set of tasks done by each resource
        resources_tasks = [model.set(nb_tasks) for r in range(nb_resources)]
        resources = model.array(resources_tasks)

        # All tasks must be sheduled on a resource
        model.constraint(model.partition(resources))

        # Creates Hexaly arrays to allow access through "at" operator
        tasks_array = model.array(tasks)
        state_array = model.array(state)
        delay_array = model.array(delay_state)

        # Makespan: end of the last task
        makespan = model.max([model.end(tasks[i]) for i in range(nb_tasks)])

        # Resource constraints
        for r in range(nb_resources):
            capacity_respected = model.lambda_function(
                lambda t: model.sum(resources[r], model.lambda_function(
                    lambda i: model.contains(tasks_array[i], t)))
                <= capacity[r])
            model.constraint(model.and_(model.range(makespan), capacity_respected))

        # State incompatibility constraints
        for r in range(nb_resources):
            state_respected = model.lambda_function(
                lambda i: model.and_(resources_tasks[r], model.lambda_function(
                    lambda j: model.or_(state_array[i]==state_array[j],
                                        (model.end(tasks_array[i]) + delay_array[state_array[i]][state_array[j]]) <=model.start(tasks_array[j]),
                                        (model.end(tasks_array[j]) + delay_array[state_array[j]][state_array[i]]) <=model.start(tasks_array[i])))))
            model.constraint(model.and_(resources_tasks[r], state_respected))

        # Minimize the makespan
        model.minimize(makespan)

        model.close()

        # Parameterize the optimizer
        optimizer.param.time_limit = time_limit

        optimizer.solve()        

        # Write the solution in a file with the following format:
        #  - total makespan, number of Tasks
        #  - for each task, the task id, the start, the end times and the ressource of the task*/
        #
        if output_file != None:
            task_resources = [0 for i in range(nb_tasks)]

            for r in range(nb_resources):
                for task in resources_tasks[r].value:
                    task_resources[task]= r

            with open(output_file, "w") as f:
                print("Solution written in file", output_file)
                f.write(str(makespan.value) + " " + str(nb_tasks) + "\n")
                for i in range(nb_tasks):
                    f.write(str(i + 1) + " " + str(tasks[i].value.start()) + " " + str(tasks[i].value.end()) + " " + str(task_resources[i]))
                    f.write("\n")


if __name__ == '__main__':
    if len(sys.argv) < 2:
        print("Usage: python rcpsp.py instance_file [output_file] [time_limit]")
        sys.exit(1)

    instance_file = sys.argv[1]
    output_file = sys.argv[2] if len(sys.argv) >= 3 else None
    time_limit = int(sys.argv[3]) if len(sys.argv) >= 4 else 60
    main(instance_file, output_file, time_limit)