import java.util.*;
import java.io.*;
import com.hexaly.optimizer.*;

public class Irp {
    // Hexaly Optimizer
    private final HexalyOptimizer optimizer;

    // Number of customers
    private int nbCustomers;

    // Horizon length
    private int horizonLength;

    // Capacity
    private int capacity;

    // Start level at the supplier
    private int startLevelSupplier;

    // Production rate of the supplier
    private int productionRateSupplier;

    // Holding costs of the supplier
    private double holdingCostSupplier;

    // Start level of the customers
    private int[] startLevel;

    // Max level of the customers
    private int[] maxLevel;

    // Demand rate of the customers
    private int[] demandRate;

    // Holding costs of the customers
    private double[] holdingCost;

    // Distance matrix
    private long[][] distMatrixData;

    // Distances between customers and supplier
    private long[] distSupplierData;

    // Decision variables
    private HxExpression[][] delivery;

    // Decision variables
    private HxExpression[] route;

    // Are the customers receiving products
    private HxExpression[][] isDelivered;

    // Distance traveled by each truck
    private HxExpression[] distRoutes;

    // Inventory at the supplier
    private HxExpression[] inventorySupplier;

    // Inventory at the customers
    private HxExpression[][] inventory;

    // Total inventory cost at the supplier
    private HxExpression totalCostInventorySupplier;

    // Inventory cost at a customer
    private HxExpression[] costInventoryCustomer;

    // Total inventory cost at customers
    private HxExpression totalCostInventory;

    // Total transportation cost
    private HxExpression totalCostRoute;

    // Objective
    private HxExpression objective;

    private Irp(HexalyOptimizer optimizer) {
        this.optimizer = optimizer;
    }

    private void solve(int limit) {
        // Declare the optimization model
        HxModel model = optimizer.getModel();

        delivery = new HxExpression[horizonLength][nbCustomers];
        route = new HxExpression[horizonLength];
        isDelivered = new HxExpression[horizonLength][nbCustomers];
        distRoutes = new HxExpression[horizonLength];

        // Quantity of product delivered at each discrete time instant of
        // the planning time horizon to each customer
        for (int t = 0; t < horizonLength; ++t) {
            for (int i = 0; i < nbCustomers; ++i) {
                delivery[t][i] = model.floatVar(0, capacity);
            }
        }

        // Sequence of customers visited at each discrete time instant of
        // the planning time horizon
        for (int t = 0; t < horizonLength; ++t) {
            route[t] = model.listVar(nbCustomers);
        }

        // Create distances as arrays to be able to access them with an "at" operator
        HxExpression distSupplier = model.array(distSupplierData);
        HxExpression distMatrix = model.array(distMatrixData);

        for (int t = 0; t < horizonLength; ++t) {
            HxExpression sequence = route[t];
            HxExpression c = model.count(sequence);

            // Customers receive products only if they are visited
            for (int i = 0; i < nbCustomers; ++i) {
                isDelivered[t][i] = model.contains(sequence, i);
            }

            // Distance traveled at instant t
            HxExpression distLambda = model
                .lambdaFunction(i -> model.at(distMatrix, model.at(sequence, model.sub(i, 1)), model.at(sequence, i)));
            distRoutes[t] = model.iif(model.gt(c, 0),
                model.sum(
                    model.sum(model.at(distSupplier, model.at(sequence, 0)), model.sum(model.range(1, c), distLambda)),
                    model.at(distSupplier, model.at(sequence, model.sub(c, 1)))),
                0);
        }

        // Stockout constraints at the supplier
        inventorySupplier = new HxExpression[horizonLength + 1];
        inventorySupplier[0] = model.createConstant(startLevelSupplier);
        for (int t = 0; t < horizonLength; ++t) {
            inventorySupplier[t + 1] = model.sum(model.sub(inventorySupplier[t], model.sum(delivery[t])),
                productionRateSupplier);
            model.constraint(model.geq(inventorySupplier[t], model.sum(delivery[t])));
        }

        // Stockout constraints at the customers
        inventory = new HxExpression[nbCustomers][horizonLength + 1];
        for (int i = 0; i < nbCustomers; ++i) {
            inventory[i][0] = model.createConstant(startLevel[i]);
            for (int t = 0; t < horizonLength; ++t) {
                inventory[i][t + 1] = model.sub(model.sum(inventory[i][t], delivery[t][i]), demandRate[i]);
                model.constraint(model.geq(inventory[i][t + 1], 0));
            }
        }

        for (int t = 0; t < horizonLength; ++t) {
            // Capacity constraints
            model.constraint(model.leq(model.sum(delivery[t]), capacity));

            // Maximum level constraints
            for (int i = 0; i < nbCustomers; ++i) {
                model.constraint(model.leq(delivery[t][i], model.sub(maxLevel[i], inventory[i][t])));
                model.constraint(model.leq(delivery[t][i], model.prod(maxLevel[i], isDelivered[t][i])));
            }
        }

        // Total inventory cost at the supplier
        totalCostInventorySupplier = model.prod(holdingCostSupplier, model.sum(inventorySupplier));

        // Total inventory cost at customers
        costInventoryCustomer = new HxExpression[nbCustomers];
        for (int i = 0; i < nbCustomers; ++i) {
            costInventoryCustomer[i] = model.prod(holdingCost[i], model.sum(inventory[i]));
        }
        totalCostInventory = model.sum(costInventoryCustomer);

        // Total transportation cost
        totalCostRoute = model.sum(distRoutes);

        // Objective: minimize the sum of all costs
        objective = model.sum(model.sum(totalCostInventorySupplier, totalCostInventory), totalCostRoute);
        model.minimize(objective);

        model.close();

        // Parametrize the optimizer
        optimizer.getParam().setTimeLimit(limit);

        optimizer.solve();
    }

    /* Write the solution in a file with the following format:
     * - total distance run by the vehicle
     * - the nodes visited at each time step (omitting the start/end at the supplier) */
    private void writeSolution(String fileName) throws IOException {
        try (PrintWriter output = new PrintWriter(fileName)) {
            output.println(totalCostRoute.getValue());
            for (int t = 0; t < horizonLength; ++t) {
                HxCollection routeCollection = route[t].getCollectionValue();
                for (int i = 0; i < routeCollection.count(); ++i) {
                    output.print((routeCollection.get(i) + 1) + " ");
                }
                output.println();
            }
        }
    }

    // The input files follow the "Archetti" format
    private void readInstance(String fileName) throws IOException {
        try (Scanner input = new Scanner(new File(fileName))) {
            input.useLocale(Locale.ROOT);
            String[] splitted;
            splitted = input.nextLine().split("\\s+");
            nbCustomers = Integer.parseInt(splitted[0]) - 1;
            horizonLength = Integer.parseInt(splitted[1]);
            capacity = Integer.parseInt(splitted[2]);
            splitted = input.nextLine().split("\\s+");
            double xCoordSupplier = Double.parseDouble(splitted[2]);
            double yCoordSupplier = Double.parseDouble(splitted[3]);
            startLevelSupplier = Integer.parseInt(splitted[4]);
            productionRateSupplier = Integer.parseInt(splitted[5]);
            holdingCostSupplier = Double.parseDouble(splitted[6]);
            double[] xCoord = new double[nbCustomers];
            double[] yCoord = new double[nbCustomers];
            startLevel = new int[nbCustomers];
            maxLevel = new int[nbCustomers];
            demandRate = new int[nbCustomers];
            holdingCost = new double[nbCustomers];
            for (int i = 0; i < nbCustomers; ++i) {
                splitted = input.nextLine().split("\\s+");
                xCoord[i] = Double.parseDouble(splitted[2]);
                yCoord[i] = Double.parseDouble(splitted[3]);
                startLevel[i] = Integer.parseInt(splitted[4]);
                maxLevel[i] = Integer.parseInt(splitted[5]);
                demandRate[i] = Integer.parseInt(splitted[7]);
                holdingCost[i] = Double.parseDouble(splitted[8]);
            }

            computeDistanceMatrix(xCoordSupplier, yCoordSupplier, xCoord, yCoord);
        }
    }

    // Compute the distance matrix
    private void computeDistanceMatrix(double xCoordSupplier, double yCoordSupplier, double[] xCoord, double[] yCoord) {
        distMatrixData = new long[nbCustomers][nbCustomers];
        for (int i = 0; i < nbCustomers; ++i) {
            distMatrixData[i][i] = 0;
            for (int j = i + 1; j < nbCustomers; ++j) {
                long dist = computeDist(xCoord[i], xCoord[j], yCoord[i], yCoord[j]);
                distMatrixData[i][j] = dist;
                distMatrixData[j][i] = dist;
            }
        }

        distSupplierData = new long[nbCustomers];
        for (int i = 0; i < nbCustomers; ++i) {
            distSupplierData[i] = computeDist(xCoordSupplier, xCoord[i], yCoordSupplier, yCoord[i]);
        }
    }

    private long computeDist(double xi, double xj, double yi, double yj) {
        return Math.round(Math.sqrt(Math.pow(xi - xj, 2) + Math.pow(yi - yj, 2)));
    }

    public static void main(String[] args) {
        if (args.length < 1) {
            System.err.println("Usage: java Irp inputFile [outputFile] [timeLimit]");
            System.exit(1);
        }

        try (HexalyOptimizer optimizer = new HexalyOptimizer()) {
            String instanceFile = args[0];
            String outputFile = args.length > 1 ? args[1] : null;
            String strTimeLimit = args.length > 2 ? args[2] : "20";

            Irp model = new Irp(optimizer);
            model.readInstance(instanceFile);
            model.solve(Integer.parseInt(strTimeLimit));
            if (outputFile != null) {
                model.writeSolution(outputFile);
            }
        } catch (Exception ex) {
            System.err.println(ex);
            ex.printStackTrace();
            System.exit(1);
        }
    }
}
