import java.util.*;
import java.io.*;
import com.hexaly.optimizer.*;

public class Kmeans {
    // Data properties
    private int nbObservations;
    private int nbDimensions;
    private int k;

    private double[][] coordinatesData;

    // Hexaly Optimizer
    private final HexalyOptimizer optimizer;

    // Decisions
    private HxExpression[] clusters;

    // Objective
    private HxExpression obj;

    private Kmeans(HexalyOptimizer optimizer) {
        this.optimizer = optimizer;
    }

    // Read instance data
    private void readInstance(int k, String fileName) throws IOException {
        try (Scanner input = new Scanner(new File(fileName))) {
            input.useLocale(Locale.ROOT);

            nbObservations = input.nextInt();
            nbDimensions = input.nextInt();

            this.k = k;
            coordinatesData = new double[nbObservations][nbDimensions];
            for (int o = 0; o < nbObservations; ++o) {
                for (int d = 0; d < nbDimensions; ++d) {
                    coordinatesData[o][d] = input.nextDouble();
                }
                input.next(); // skip initial clusters
            }
        }
    }

    private void solve(int limit) {
        // Declare the optimization model
        HxModel model = optimizer.getModel();

        // Set decisions: clusters[c] represents the points in cluster c
        clusters = new HxExpression[k];
        for (int c = 0; c < k; ++c) {
            clusters[c] = model.setVar(nbObservations);
        }

        // Each point must be in one cluster and one cluster only
        model.constraint(model.partition(clusters));

        // Coordinates of points
        HxExpression coordinates = model.array(coordinatesData);

        // Compute variances
        HxExpression[] variances = new HxExpression[k];
        for (int c = 0; c < k; ++c) {
            HxExpression cluster = clusters[c];
            HxExpression size = model.count(cluster);

            // Compute the centroid of the cluster
            HxExpression centroid = model.array();
            for (int d = 0; d < nbDimensions; ++d) {
                HxExpression dExpr = model.createConstant(d);
                HxExpression coordinateLambda = model.lambdaFunction(o -> model.at(coordinates, o, dExpr));
                centroid
                    .addOperand(model.iif(model.eq(size, 0), 0, model.div(model.sum(cluster, coordinateLambda), size)));
            }

            // Compute the variance of the cluster
            HxExpression variance = model.sum();
            for (int d = 0; d < nbDimensions; ++d) {
                HxExpression dExpr = model.createConstant(d);
                HxExpression dimensionVarianceLambda = model.lambdaFunction(
                    o -> model.pow(model.sub(model.at(coordinates, o, dExpr), model.at(centroid, dExpr)), 2));
                HxExpression dimensionVariance = model.sum(cluster, dimensionVarianceLambda);
                variance.addOperand(dimensionVariance);
            }
            variances[c] = variance;
        }

        // Minimize the total variance
        obj = model.sum(variances);
        model.minimize(obj);

        model.close();

        // Parametrize the optimizer
        optimizer.getParam().setTimeLimit(limit);

        optimizer.solve();
    }

    /*
     * Write the solution in a file in the following format:
     * - objective value
     * - k
     * - for each cluster, a line with the elements in the cluster (separated by
     * spaces)
     */
    private void writeSolution(String fileName) throws IOException {
        try (PrintWriter output = new PrintWriter(fileName)) {
            output.println(obj.getDoubleValue());
            output.println(k);
            for (int c = 0; c < k; ++c) {
                HxCollection clusterCollection = clusters[c].getCollectionValue();
                for (int i = 0; i < clusterCollection.count(); ++i) {
                    output.print(clusterCollection.get(i) + " ");
                }
                output.println();
            }
        }
    }

    public static void main(String[] args) {
        if (args.length < 1) {
            System.err.println("Usage: java Kmeans inputFile [outputFile] [timeLimit] [k value]");
            System.exit(1);
        }

        String instanceFile = args[0];
        String outputFile = args.length > 1 ? args[1] : null;
        String strTimeLimit = args.length > 2 ? args[2] : "5";
        String k = args.length > 3 ? args[3] : "2";

        try (HexalyOptimizer optimizer = new HexalyOptimizer()) {
            Kmeans model = new Kmeans(optimizer);
            model.readInstance(Integer.parseInt(k), instanceFile);
            model.solve(Integer.parseInt(strTimeLimit));
            if (outputFile != null) {
                model.writeSolution(outputFile);
            }
        } catch (Exception ex) {
            System.err.println(ex);
            ex.printStackTrace();
            System.exit(1);
        }
    }
}
