import java.util.*;
import java.io.*;
import com.hexaly.optimizer.*;

public class LocationRoutingProblem {
    // Hexaly Optimizer
    private final HexalyOptimizer optimizer;

    // Number of customers
    private int nbCustomers;

    // Customers coordinates
    private int[] xCustomers;
    private int[] yCustomers;

    // Customers demands
    private int[] demandsData;

    // Number of depots
    private int nbDepots;

    // Depots coordinates
    private int[] xDepots;
    private int[] yDepots;

    // Capacity of depots
    private int[] depotsCapacity;

    // Cost of opening a depot
    private double[] openingDepotsCost;

    // Number of trucks
    private int nbTrucks;

    // Capacity of trucks
    private int truckCapacity;

    // Cost of opening a route
    private int openingRouteCost;

    // Is the sequence used ?
    private HxExpression[] sequenceUsed;

    // What is the depot of the sequence ?
    private HxExpression[] associatedDepot;

    // Distance matrixes
    private double[][] distMatrixData;
    private double[][] distDepotsData;

    private int areCostDouble;

    // Decision variables
    private HxExpression[] customersSequences;
    private HxExpression[] depots;

    // Objective value
    private HxExpression totalCost;

    private LocationRoutingProblem(HexalyOptimizer optimizer) {
        this.optimizer = optimizer;
    }

    private void solve(int limit) {
        // Declare the optimization model
        HxModel m = optimizer.getModel();

        int totalDemand = 0;
        for (int d : demandsData) {
            totalDemand += d;
        }
        int minNbTrucks = (int) Math.ceil(totalDemand / truckCapacity);
        int nbTrucks = (int) Math.ceil(1.5 * minNbTrucks);

        customersSequences = new HxExpression[nbTrucks];
        depots = new HxExpression[nbDepots];

        // A route is represented as a list containing the customers in the order they are visited
        for (int i = 0; i < nbTrucks; ++i) {
            customersSequences[i] = m.listVar(nbCustomers);
        }
        // All customers should be assigned to a route
        m.constraint(m.partition(customersSequences));
        // A depot is represented as a set containing the associated customersSequences
        for (int d = 0; d < nbDepots; ++d) {
            depots[d] = m.setVar(nbTrucks);
        }
        // All the customersSequences should be assigned to a depot
        m.constraint(m.partition(depots));

        HxExpression demands = m.array(demandsData);
        HxExpression distMatrix = m.array(distMatrixData);
        HxExpression distDepots = m.array(distDepotsData);
        sequenceUsed = new HxExpression[nbTrucks];
        HxExpression[] routeCosts = new HxExpression[nbTrucks];
        HxExpression[] distRoutes = new HxExpression[nbTrucks];
        associatedDepot = new HxExpression[nbTrucks];
        HxExpression quantityServed = m.array();

        for (int r = 0; r < nbTrucks; ++r) {
            HxExpression sequence = customersSequences[r];
            HxExpression c = m.count(sequence);

            // A sequence is used if it serves at least one customer
            sequenceUsed[r] = m.gt(c, 0);
            // The "find" function gets the depot that is assigned to the sequence
            associatedDepot[r] = m.find(m.array(depots), r);

            HxExpression demandLambda = m.lambdaFunction(j -> m.at(demands, j));
            quantityServed.addOperand(m.sum(sequence, demandLambda));
            // The quantity needed in each sequence must not exceed the vehicle capacity
            m.constraint(m.leq(m.at(quantityServed, r), truckCapacity));

            HxExpression distLambda = m
                .lambdaFunction(i -> m.at(distMatrix, m.at(sequence, i), m.at(sequence, m.sum(i, 1))));

            distRoutes[r] = m.sum(m.sum(m.range(0, m.sub(c, 1)), distLambda),
                m.iif(sequenceUsed[r], m.sum(m.at(distDepots, m.at(sequence, 0), associatedDepot[r]),
                    m.at(distDepots, m.at(sequence, m.sub(c, 1)), associatedDepot[r])), 0));
            // The sequence cost is the sum of the opening cost and the sequence length
            routeCosts[r] = m.sum(distRoutes[r], m.prod(openingRouteCost, sequenceUsed[r]));
        }

        HxExpression[] depotCost = new HxExpression[nbDepots];
        for (int d = 0; d < nbDepots; ++d) {
            // A depot is open if at least a sequence starts from there
            depotCost[d] = m.prod(openingDepotsCost[d], m.gt(m.count(depots[d]), 0));

            HxExpression depotLambda = m.lambdaFunction(r -> m.at(quantityServed, r));
            HxExpression depotQuantity = m.sum(depots[d], depotLambda);
            // The total demand served by a depot must not exceed its capacity
            m.constraint(m.leq(depotQuantity, depotsCapacity[d]));
        }
        HxExpression depotsCost = m.sum(depotCost);
        HxExpression routingCost = m.sum(routeCosts);
        totalCost = m.sum(routingCost, depotsCost);

        m.minimize(totalCost);
        m.close();

        optimizer.getParam().setTimeLimit(limit);
        optimizer.solve();
    }

    /* Write the solution in a file */
    private void writeSolution(String infile, String outfile) throws IOException {
        try (PrintWriter output = new PrintWriter(outfile)) {
            output.println("File name: " + infile + "; total cost = " + totalCost.getDoubleValue());
            for (int r = 0; r < nbTrucks; ++r) {
                if (sequenceUsed[r].getIntValue() != 0) {
                    output.print("Route " + r + ", assigned to depot " + associatedDepot[r].getIntValue() + " : ");
                    HxCollection customersCollection = customersSequences[r].getCollectionValue();
                    for (int i = 0; i < customersCollection.count(); ++i) {
                        output.print(customersCollection.get(i) + " ");
                    }
                    output.println();
                }
            }
        } catch (FileNotFoundException e) {
            System.out.println("An error occurred.");
            e.printStackTrace();
        }
    }

    private void readInstanceLrp(String fileName) throws IOException {
        try (Scanner input = new Scanner(new File(fileName))) {
            input.useLocale(Locale.ROOT);
            nbCustomers = input.nextInt();
            xCustomers = new int[nbCustomers];
            yCustomers = new int[nbCustomers];
            demandsData = new int[nbCustomers];
            distMatrixData = new double[nbCustomers][];
            distDepotsData = new double[nbCustomers][];
            nbDepots = input.nextInt();
            xDepots = new int[nbDepots];
            yDepots = new int[nbDepots];
            depotsCapacity = new int[nbDepots];
            openingDepotsCost = new double[nbDepots];

            for (int i = 0; i < nbDepots; ++i) {
                xDepots[i] = input.nextInt();
                yDepots[i] = input.nextInt();
            }
            for (int i = 0; i < nbCustomers; ++i) {
                xCustomers[i] = input.nextInt();
                yCustomers[i] = input.nextInt();
            }
            truckCapacity = input.nextInt();
            for (int i = 0; i < nbDepots; ++i) {
                depotsCapacity[i] = input.nextInt();
            }
            for (int i = 0; i < nbCustomers; ++i) {
                demandsData[i] = input.nextInt();
            }
            double[] tempOpeningCostDepots = new double[nbDepots];
            for (int i = 0; i < nbDepots; ++i) {
                if (input.hasNext()) {
                    tempOpeningCostDepots[i] = Double.parseDouble(input.next());
                } else if (input.hasNextInt()) {
                    tempOpeningCostDepots[i] = input.nextInt();
                }
            }
            int tempOpeningCostRoute = input.nextInt();
            areCostDouble = input.nextInt();
            if (areCostDouble == 1) {
                openingRouteCost = tempOpeningCostRoute;
                for (int i = 0; i < tempOpeningCostDepots.length; ++i) {
                    openingDepotsCost[i] = tempOpeningCostDepots[i];
                }
            } else {
                openingRouteCost = Math.round(tempOpeningCostRoute);
                for (int i = 0; i < tempOpeningCostDepots.length; ++i) {
                    openingDepotsCost[i] = Math.round(tempOpeningCostDepots[i]);
                }
            }
            computeDistanceMatrix();
        } catch (FileNotFoundException e) {
            System.out.println("An error occurred.");
            e.printStackTrace();
        }
    }

    void computeDistanceMatrix() {
        for (int i = 0; i < nbCustomers; ++i) {
            distMatrixData[i] = new double[nbCustomers];
            for (int j = 0; j < nbCustomers; ++j) {
                distMatrixData[i][j] = computeDist(xCustomers[i], xCustomers[j], yCustomers[i], yCustomers[j],
                    areCostDouble);
            }
            distDepotsData[i] = new double[nbDepots];
            for (int d = 0; d < nbDepots; ++d) {
                distDepotsData[i][d] = computeDist(xCustomers[i], xDepots[d], yCustomers[i], yDepots[d], areCostDouble);
            }
        }
    }

    private double computeDist(int xi, int xj, int yi, int yj, int areCostDouble) {
        double dist = Math.sqrt(Math.pow(xi - xj, 2) + Math.pow(yi - yj, 2));
        if (areCostDouble == 0) {
            dist = Math.ceil(100 * dist);
        }
        return dist;
    }

    public static void main(String[] args) {
        if (args.length < 1) {
            System.err.println("Usage: java LocationRoutingProblem inputFile [outputFile] [timeLimit]");
            System.exit(1);
        }

        try (HexalyOptimizer optimizer = new HexalyOptimizer()) {
            String instanceFile = args[0];
            String strOutfile = args.length > 1 ? args[1] : null;
            String strTimeLimit = args.length > 2 ? args[2] : "20";

            LocationRoutingProblem model = new LocationRoutingProblem(optimizer);
            model.readInstanceLrp(instanceFile);
            model.solve(Integer.parseInt(strTimeLimit));
            if (strOutfile != null)
                model.writeSolution(instanceFile, strOutfile);
        } catch (Exception ex) {
            System.err.println(ex);
            ex.printStackTrace();
            System.exit(1);
        }
    }
}
