using System;
using System.IO;
using Hexaly.Optimizer;

public class Maxcut : IDisposable
{
    // Number of vertices
    int n;

    // Number of edges
    int m;

    // Origin of each edge
    int[] origin;

    // Destination of each edge
    int[] dest;

    // Weight of each edge
    int[] w;

    // Hexaly Optimizer
    HexalyOptimizer optimizer;

    // True if vertex x[i] is on the right side of the cut
    // and false if it is on the left side of the cut
    HxExpression[] x;

    // Objective
    HxExpression cutWeight;

    public Maxcut()
    {
        optimizer = new HexalyOptimizer();
    }

    /* Read instance data */
    public void ReadInstance(string fileName)
    {
        using (StreamReader input = new StreamReader(fileName))
        {
            var tokens = input.ReadLine().Split(' ');
            n = int.Parse(tokens[0]);
            m = int.Parse(tokens[1]);

            origin = new int[m];
            dest = new int[m];
            w = new int[m];

            for (int e = 0; e < m; ++e)
            {
                tokens = input.ReadLine().Split(' ');
                origin[e] = int.Parse(tokens[0]);
                dest[e] = int.Parse(tokens[1]);
                w[e] = int.Parse(tokens[2]);
            }
        }
    }

    public void Dispose()
    {
        if (optimizer != null)
            optimizer.Dispose();
    }

    public void Solve(int limit)
    {
        // Declare the optimization model
        HxModel model = optimizer.GetModel();

        // Decision variables x[i]
        x = new HxExpression[n];
        for (int i = 0; i < n; ++i)
            x[i] = model.Bool();

        // An edge is in the cut-set if it has an extremity in each class of the bipartition
        // Note: the indices start at 1 in the instances
        HxExpression[] incut = new HxExpression[m];
        for (int e = 0; e < m; ++e)
            incut[e] = model.Neq(x[origin[e] - 1], x[dest[e] - 1]);

        // Size of the cut
        cutWeight = model.Sum();
        for (int e = 0; e < m; ++e)
            cutWeight.AddOperand(w[e] * incut[e]);
        model.Maximize(cutWeight);

        model.Close();

        // Parametrize the optimizer
        optimizer.GetParam().SetTimeLimit(limit);
        optimizer.Solve();
    }

    /* Write the solution in a file with the following format:
     *  - objective value
     *  - each line contains a vertex number and its subset (1 for S, 0 for V-S) */
    public void WriteSolution(string fileName)
    {
        using (StreamWriter output = new StreamWriter(fileName))
        {
            output.WriteLine(cutWeight.GetValue());
            for (int i = 0; i < n; ++i)
                output.WriteLine((i + 1) + " " + x[i].GetValue());
        }
    }

    public static void Main(string[] args)
    {
        if (args.Length < 1)
        {
            Console.WriteLine("Usage: Maxcut inputFile [solFile] [timeLimit]");
            Environment.Exit(1);
        }
        string instanceFile = args[0];
        string outputFile = args.Length > 1 ? args[1] : null;
        string strTimeLimit = args.Length > 2 ? args[2] : "10";

        using (Maxcut model = new Maxcut())
        {
            model.ReadInstance(instanceFile);
            model.Solve(int.Parse(strTimeLimit));
            if (outputFile != null)
                model.WriteSolution(outputFile);
        }
    }
}
