import java.util.*;
import java.io.*;
import com.hexaly.optimizer.*;

public class Mdvrp {
    // Hexaly Optimizer
    private final HexalyOptimizer optimizer;

    // Number of customers
    int nbCustomers;

    // Number of depots/warehouses
    int nbDepots;

    // Number of trucks per depot
    private int nbTrucksPerDepot;

    // Capacity of the trucks per depot
    private long[] truckCapacity;

    // Duration capacity of the trucks per depot
    private long[] routeDurationCapacity;

    // Service time per customer
    private long[] serviceTimeData;

    // Demand per customer
    private long[] demandsData;

    // Distance matrix between customers
    private double[][] distanceMatrixCustomersData;

    // Distance matrix between customers and depots
    private double[][] distanceWarehouseData;

    // Decision variable
    private HxExpression[][] customersSequences;

    // Distance traveled by all the trucks
    private HxExpression totalDistance;

    private Mdvrp(HexalyOptimizer optimizer) {
        this.optimizer = optimizer;
    }

    private void readInstance(String fileName) throws IOException {
        readInputMdvrp(fileName);
    }

    private void solve(int limit) {
        // Declare the optimization model
        HxModel model = optimizer.getModel();

        // Sequence of customers visited by each truck
        customersSequences = new HxExpression[nbDepots][nbTrucksPerDepot];

        // Vectorization for partition constraint
        HxExpression[] customersSequencesConstraint = new HxExpression[nbDepots * nbTrucksPerDepot];

        // Sequence of customers visited by each truck
        for (int d = 0; d < nbDepots; ++d) {
            for (int k = 0; k < nbTrucksPerDepot; ++k) {
                customersSequences[d][k] = model.listVar(nbCustomers);
                customersSequencesConstraint[d * nbTrucksPerDepot + k] = customersSequences[d][k];
            }
        }
        // All customers must be visited by exactly one truck
        model.constraint(model.partition(customersSequencesConstraint));

        // Create HexalyOptimizer arrays to be able to access them with an "at" operator
        HxExpression demands = model.array(demandsData);
        HxExpression serviceTime = model.array(serviceTimeData);

        HxExpression distMatrix = model.array(distanceMatrixCustomersData);

        // Distances for each truck from each depot
        HxExpression[][] routeDistances = new HxExpression[nbDepots][nbTrucksPerDepot];

        // Total distance traveled
        totalDistance = model.sum();

        for (int d = 0; d < nbDepots; ++d) {
            HxExpression distDepot = model.array(distanceWarehouseData[d]);

            for (int k = 0; k < nbTrucksPerDepot; ++k) {
                HxExpression sequence = customersSequences[d][k];
                HxExpression c = model.count(sequence);

                // The quantity needed in each route must not exceed the truck capacity
                HxExpression demandLambda = model.lambdaFunction(j -> model.at(demands, j));
                HxExpression routeQuantity = model.sum(sequence, demandLambda);
                model.constraint(model.leq(routeQuantity, truckCapacity[d]));

                // Distance traveled by truck k of depot d
                HxExpression distLambda = model
                        .lambdaFunction(
                                i -> model.at(distMatrix, model.at(sequence, model.sub(i, 1)), model.at(sequence, i)));
                routeDistances[d][k] = model.sum(model.sum(model.range(1, c), distLambda),
                        model.iif(model.gt(c, 0), model.sum(model.at(distDepot, model.at(sequence, 0)),
                                model.at(distDepot, model.at(sequence, model.sub(c, 1)))), 0));

                totalDistance.addOperand(routeDistances[d][k]);

                // We add service time
                HxExpression serviceLambda = model.lambdaFunction(j -> model.at(serviceTime, j));
                HxExpression routeServiceTime = model.sum(sequence, serviceLambda);

                // The total distance should not exceed the duration capacity of the truck (only
                // if we define such a capacity)
                if (routeDurationCapacity[d] > 0) {
                    model.constraint(
                            model.leq(model.sum(routeDistances[d][k], routeServiceTime), routeDurationCapacity[d]));
                }
            }
        }

        // Objective: minimize the total distance traveled
        model.minimize(totalDistance);

        model.close();

        // Parametrize the optimizer
        optimizer.getParam().setTimeLimit(limit);

        optimizer.solve();
    }

    // Write the solution in a file with the following format:
    // - instance, time_limit, total distance
    // - for each depot and for each truck in this depot, the customers visited
    private void writeSolution(String fileName, String instanceFile, String timeLimit) throws IOException {
        try (PrintWriter outfile = new PrintWriter(fileName)) {
            outfile.println("Instance: " + instanceFile + " ; time_limit: " + timeLimit + " ; Objective value: "
                    + totalDistance.getDoubleValue());
            for (int d = 0; d < nbDepots; ++d) {
                ArrayList<Integer> trucksUsed = new ArrayList<Integer>();
                for (int k = 0; k < nbTrucksPerDepot; k++) {
                    if (customersSequences[d][k].getCollectionValue().count() > 0) {
                        trucksUsed.add(k);
                    }
                }

                if (trucksUsed.size() > 0) {
                    outfile.println("Depot " + (d + 1));
                    for (int k = 0; k < trucksUsed.size(); ++k) {
                        outfile.print("Truck " + (k + 1) + " : ");
                        HxCollection customersCollection = customersSequences[d][trucksUsed.get(k)]
                                .getCollectionValue();
                        for (int p = 0; p < customersCollection.count(); ++p)
                            outfile.print((customersCollection.get(p) + 1) + " ");
                        outfile.println();
                    }
                    outfile.println();
                }
            }
        }
    }

    // Input files following "Cordeau"'s format
    private void readInputMdvrp(String fileName) throws IOException {
        try (Scanner input = new Scanner(new File(fileName))) {
            input.useLocale(Locale.ROOT);
            String[] splitted;

            splitted = input.nextLine().split(" ");

            // Numbers of trucks per depot, customers and depots
            nbTrucksPerDepot = Integer.parseInt(splitted[1].trim());
            nbCustomers = Integer.parseInt(splitted[2].trim());
            nbDepots = Integer.parseInt(splitted[3].trim());

            routeDurationCapacity = new long[nbDepots];
            truckCapacity = new long[nbDepots];

            for (int d = 0; d < nbDepots; ++d) {
                routeDurationCapacity[d] = input.nextInt();
                truckCapacity[d] = input.nextInt();
            }

            // Coordinates X and Y, service time and demand for customers
            double[] nodesX = new double[nbCustomers];
            double[] nodesY = new double[nbCustomers];
            serviceTimeData = new long[nbCustomers];
            demandsData = new long[nbCustomers];

            for (int n = 0; n < nbCustomers; ++n) {
                input.nextLine();
                input.nextInt();

                nodesX[n] = input.nextDouble();
                nodesY[n] = input.nextDouble();
                serviceTimeData[n] = input.nextInt();
                demandsData[n] = input.nextInt();
            }

            // Coordinates X and Y for depots
            double[] DepotX = new double[nbDepots];
            double[] DepotY = new double[nbDepots];

            for (int d = 0; d < nbDepots; ++d) {
                input.nextLine();
                input.nextInt();
                DepotX[d] = input.nextDouble();
                DepotY[d] = input.nextDouble();
            }

            // Compute the distance matrices
            ComputeDistanceMatrixCustomers(nodesX, nodesY);
            ComputeDistanceWarehouse(DepotX, DepotY, nodesX, nodesY);
        }
    }

    // Compute the distance matrix for customers
    private void ComputeDistanceMatrixCustomers(double[] nodesX, double[] nodesY) {
        distanceMatrixCustomersData = new double[nbCustomers][nbCustomers];

        for (int i = 0; i < nbCustomers; ++i) {
            distanceMatrixCustomersData[i][i] = 0;
            for (int j = i + 1; j < nbCustomers; ++j) {
                double dist = computeDist(nodesX[i], nodesX[j], nodesY[i], nodesY[j]);
                distanceMatrixCustomersData[i][j] = dist;
                distanceMatrixCustomersData[j][i] = dist;
            }
        }
    }

    // Compute the distance matrix for depots/warehouses
    private void ComputeDistanceWarehouse(double[] DepotX, double[] DepotY, double[] nodesX, double[] nodesY) {
        distanceWarehouseData = new double[nbDepots][nbCustomers];

        for (int d = 0; d < nbDepots; ++d) {
            for (int i = 0; i < nbCustomers; ++i) {
                distanceWarehouseData[d][i] = computeDist(nodesX[i], DepotX[d], nodesY[i], DepotY[d]);
            }
        }
    }

    private double computeDist(double xi, double xj, double yi, double yj) {
        return Math.sqrt(Math.pow(xi - xj, 2) + Math.pow(yi - yj, 2));
    }

    public static void main(String[] args) {
        if (args.length < 1) {
            System.err.println("Usage: java Mdvrp inputFile [outputFile] [timeLimit]");
            System.exit(1);
        }
        try (HexalyOptimizer optimizer = new HexalyOptimizer()) {
            String instanceFile = args[0];
            String outputFile = args.length > 1 ? args[1] : null;
            String strTimeLimit = args.length > 2 ? args[2] : "20";

            Mdvrp model = new Mdvrp(optimizer);
            model.readInstance(instanceFile);
            model.solve(Integer.parseInt(strTimeLimit));

            if (outputFile != null) {
                model.writeSolution(outputFile, instanceFile, strTimeLimit);
            }
        } catch (Exception ex) {
            System.err.println(ex);
            ex.printStackTrace();
            System.exit(1);
        }
    }

}
