import java.util.*;
import java.io.*;
import com.hexaly.optimizer.*;
import java.lang.Math;

public class MultiTripVrp {
    // Hexaly Optimizer
    private final HexalyOptimizer optimizer;

    // Number of customers
    private int nbCustomers;

    // Number of depots
    private int nbDepots;

    // Number of depot copies
    private int nbDepotCopies;

    // Total number of locations (customers, depots, depots copies)
    private int nbTotalLocations;

    // Capacity of the trucks
    private int truckCapacity;

    // Demand on each customer
    private long[] demandsData;

    // Distance matrix between customers
    private long[][] distMatrixData;

    // Number of trucks
    private int nbTrucks;

    // Maximum distance traveled by a truck
    private int maxDist;

    // Decision variables
    private HxExpression[] visitOrders;

    // Are the trucks actually used
    private HxExpression[] trucksUsed;

    // Number of trucks used in the solution
    private HxExpression nbTrucksUsed;

    // Distance traveled by all the trucks
    private HxExpression totalDistance;

    private MultiTripVrp(HexalyOptimizer optimizer) {
        this.optimizer = optimizer;
    }

    private void solve(int limit) {
        // Declare the optimization m
        HxModel m = optimizer.getModel();

        trucksUsed = new HxExpression[nbTrucks];
        visitOrders = new HxExpression[nbTrucks + 1];
        HxExpression[] distRoutes = new HxExpression[nbTrucks];

        // Locations visited by each truck (Customers and Depots)
        // Add copies of the depots (so that they can be visited multiple times)
        // Add an extra fictive truck (who will visit every depot/ depot copy that will
        // not be visited by real trucks)
        for (int k = 0; k < nbTrucks + 1; ++k)
            visitOrders[k] = m.listVar(nbTotalLocations);

        // The fictive truck cannot visit customers
        for (int k = 0; k < nbCustomers; ++k)
            m.constraint(m.not((m.contains(visitOrders[nbTrucks], k))));

        // All customers must be visited by exactly one truck
        m.constraint(m.partition(visitOrders));

        // Create HexalyOptimizer arrays to be able to access them with an "at" operator
        HxExpression demands = m.array(demandsData);
        HxExpression distMatrix = m.array(distMatrixData);

        for (int k = 0; k < nbTrucks; ++k) {
            HxExpression sequence = visitOrders[k];
            HxExpression c = m.count(sequence);

            // A truck is used if it visits at least one customer
            trucksUsed[k] = m.gt(c, 0);
            // The quantity needed in each route must not exceed the truck capacity
            HxExpression routeQuantityLambda = m.lambdaFunction((i, prev) -> m
                    .iif(m.leq(m.at(sequence, i), nbCustomers - 1), m.sum(prev, m.at(demands, m.at(sequence, i))), 0));
            HxExpression routeQuantity = m.array(m.range(0, c), routeQuantityLambda, 0);
            // Trucks cannot carry more than their capacity
            HxExpression quantityLambda = m.lambdaFunction(i -> m.leq(m.at(routeQuantity, i), truckCapacity));
            m.constraint(m.and(m.range(0, c), quantityLambda));

            // Distance traveled by truck k
            HxExpression distLambda = m
                    .lambdaFunction(i -> m.at(distMatrix, m.at(sequence, m.sub(i, 1)), m.at(sequence, i)));
            distRoutes[k] = m.sum(m.sum(m.range(1, c), distLambda),
                    m.iif(m.gt(c, 0), m.sum(m.at(m.at(distMatrix, nbCustomers), m.at(sequence, 0)),
                            m.at(m.at(distMatrix, m.at(sequence, m.sub(c, 1))), nbCustomers)), 0));

            m.constraint(m.leq(distRoutes[k], maxDist));
        }

        totalDistance = m.sum(distRoutes);

        // Objective: minimize the distance traveled
        m.minimize(totalDistance);

        m.close();

        // Parametrize the optimizer
        optimizer.getParam().setTimeLimit(limit);

        optimizer.solve();
    }

    /* Write the solution in a file */
    private void writeSolution(String infile, String outfile) throws IOException {
        try (PrintWriter output = new PrintWriter(outfile)) {
            output.println("File name: " + infile + "; total distance = " + totalDistance.getValue());
            for (int r = 0; r < nbTrucks; ++r) {
                if (trucksUsed[r].getValue() != 0) {
                    output.print("Truck " + r + " : ");
                    HxCollection visitCollection = visitOrders[r].getCollectionValue();
                    for (int i = 0; i < visitCollection.count(); ++i) {
                        output.print(
                                (visitCollection.get(i) < nbCustomers ? visitCollection.get(i)
                                        : -((int) (Math.floor((visitCollection.get(i) - nbCustomers) / nbDepotCopies)
                                                + 1)))
                                        + " ");
                    }
                    output.println();
                }
            }
        } catch (FileNotFoundException e) {
            System.out.println("An error occurred.");
            e.printStackTrace();
        }
    }

    private void readInstanceMultiTripVrp(String fileName) throws IOException {
        try (Scanner input = new Scanner(new File(fileName))) {
            input.useLocale(Locale.ROOT);
            nbDepotCopies = 20;
            nbCustomers = input.nextInt();
            long[] customersX = new long[nbCustomers];
            long[] customersY = new long[nbCustomers];
            demandsData = new long[nbCustomers];
            nbDepots = input.nextInt();
            long[] depotsX = new long[nbDepots];
            long[] depotsY = new long[nbDepots];

            for (int i = 0; i < nbDepots; ++i) {
                depotsX[i] = input.nextInt();
                depotsY[i] = input.nextInt();
            }
            for (int i = 0; i < nbCustomers; ++i) {
                customersX[i] = input.nextInt();
                customersY[i] = input.nextInt();
            }
            truckCapacity = input.nextInt() / 2;
            for (int i = 0; i < nbDepots; ++i) {
                input.next();
            }
            for (int i = 0; i < nbCustomers; ++i) {
                demandsData[i] = input.nextInt();
            }

            nbTotalLocations = nbCustomers + nbDepots * nbDepotCopies;

            nbTrucks = 3;

            maxDist = 400;

            computeDistanceMatrix(depotsX, depotsY, customersX, customersY);

        } catch (FileNotFoundException e) {
            System.out.println("An error occurred.");
            e.printStackTrace();
        }
    }

    // Compute the distance matrix
    private void computeDistanceMatrix(long[] depotsX, long[] depotsY, long[] customersX, long[] customersY) {
        distMatrixData = new long[nbTotalLocations][nbTotalLocations];

        for (int i = 0; i < nbCustomers; ++i) {
            distMatrixData[i][i] = 0;
            for (int j = i + 1; j < nbCustomers; ++j) {
                long distance = computeDist(customersX[i], customersX[j], customersY[i], customersY[j]);
                distMatrixData[i][j] = distance;
                distMatrixData[j][i] = distance;
            }

            for (int d = 0; d < nbDepots; d++) {
                long distance = computeDist(customersX[i], depotsX[d], customersY[i], depotsY[d]);
                for (int c = 0; c < nbDepotCopies; c++) {
                    int j = nbCustomers + d * nbDepotCopies + c;
                    distMatrixData[i][j] = distance;
                    distMatrixData[j][i] = distance;
                }
            }
        }

        for (int i = nbCustomers; i < nbTotalLocations; i++) {
            for (int j = nbCustomers; j < nbTotalLocations; j++) {
                // Going from one depot to an other is never worth it
                distMatrixData[i][j] = 100000;
            }
        }
    }

    private long computeDist(long xi, long xj, long yi, long yj) {
        double exactDist = Math.sqrt(Math.pow(xi - xj, 2) + Math.pow(yi - yj, 2));
        return Math.round(exactDist);
    }

    public static void main(String[] args) {
        if (args.length < 1) {
            System.err.println("Usage: java MultiTripVrp inputFile [outputFile] [timeLimit]");
            System.exit(1);
        }

        try (HexalyOptimizer optimizer = new HexalyOptimizer()) {
            String instanceFile = args[0];
            String strOutfile = args.length > 1 ? args[1] : null;
            String strTimeLimit = args.length > 2 ? args[2] : "20";

            MultiTripVrp model = new MultiTripVrp(optimizer);
            model.readInstanceMultiTripVrp(instanceFile);
            model.solve(Integer.parseInt(strTimeLimit));
            if (strOutfile != null)
                model.writeSolution(instanceFile, strOutfile);
        } catch (Exception ex) {
            System.err.println(ex);
            ex.printStackTrace();
            System.exit(1);
        }
    }
}
