using System;
using System.IO;
using Hexaly.Optimizer;

public class OptimalBucket : IDisposable
{
    // Hexaly Optimizer
    HexalyOptimizer optimizer;

    // Hexaly Program variables
    HxExpression R;
    HxExpression r;
    HxExpression h;

    HxExpression surface;
    HxExpression volume;

    public OptimalBucket()
    {
        optimizer = new HexalyOptimizer();
    }

    public void Dispose()
    {
        if (optimizer != null)
            optimizer.Dispose();
    }

    public void Solve(int limit)
    {
        // Declare the optimization model
        HxModel model = optimizer.GetModel();

        // Numerical decisions
        R = model.Float(0, 1);
        r = model.Float(0, 1);
        h = model.Float(0, 1);

        // Surface must not exceed the surface of the plain disc
        surface =
            Math.PI * model.Pow(r, 2)
            + Math.PI * (R + r) * model.Sqrt(model.Pow(R - r, 2) + model.Pow(h, 2));
        model.AddConstraint(surface <= Math.PI);

        // Maximize the volume
        volume = Math.PI * h / 3 * (model.Pow(R, 2) + R * r + model.Pow(r, 2));
        model.Maximize(volume);

        model.Close();

        // Parametrize the optimizer
        optimizer.GetParam().SetTimeLimit(limit);

        optimizer.Solve();
    }

    // Write the solution in a file with the following format:
    //  - surface and volume of the bucket
    //  - values of R, r and h
    public void WriteSolution(string fileName)
    {
        using (StreamWriter output = new StreamWriter(fileName))
        {
            output.WriteLine(surface.GetDoubleValue() + " " + volume.GetDoubleValue());
            output.WriteLine(
                R.GetDoubleValue() + " " + r.GetDoubleValue() + " " + h.GetDoubleValue()
            );
        }
    }

    public static void Main(string[] args)
    {
        string outputFile = args.Length > 0 ? args[0] : null;
        string strTimeLimit = args.Length > 1 ? args[1] : "2";

        using (OptimalBucket model = new OptimalBucket())
        {
            model.Solve(int.Parse(strTimeLimit));
            if (outputFile != null)
                model.WriteSolution(outputFile);
        }
    }
}
