import java.io.*;
import com.hexaly.optimizer.*;

public class OptimalBucket {
    private static final double PI = 3.14159265359;

    // Hexaly Optimizer
    private final HexalyOptimizer optimizer;

    // Hexaly Program variables
    private HxExpression R;
    private HxExpression r;
    private HxExpression h;

    private HxExpression surface;
    private HxExpression volume;

    private OptimalBucket(HexalyOptimizer optimizer) {
        this.optimizer = optimizer;
    }

    private void solve(int limit) {
        // Declare the optimization model
        HxModel model = optimizer.getModel();
        HxExpression piConst = model.createConstant(PI);

        // Numerical decisions
        R = model.floatVar(0, 1);
        r = model.floatVar(0, 1);
        h = model.floatVar(0, 1);

        // surface = PI*r^2 + PI*(R+r)*sqrt((R - r)^2 + h^2)
        HxExpression s1 = model.prod(piConst, r, r);
        HxExpression s2 = model.pow(model.sub(R, r), 2);
        HxExpression s3 = model.pow(h, 2);
        HxExpression s4 = model.sqrt(model.sum(s2, s3));
        HxExpression s5 = model.sum(R, r);
        HxExpression s6 = model.prod(piConst, s5, s4);
        surface = model.sum(s1, s6);

        // Surface must not exceed the surface of the plain disc
        model.addConstraint(model.leq(surface, PI));

        HxExpression v1 = model.pow(R, 2);
        HxExpression v2 = model.prod(R, r);
        HxExpression v3 = model.pow(r, 2);

        // volume = PI*h/3*(R^2 + R*r + r^2)
        volume = model.prod(piConst, model.div(h, 3), model.sum(v1, v2, v3));

        // Maximize the volume
        model.maximize(volume);

        model.close();

        // Parametrize the optimizer
        optimizer.getParam().setTimeLimit(limit);

        optimizer.solve();
    }

    /* Write the solution in a file with the following format:
     * - surface and volume of the bucket
     * - values of R, r and h */
    private void writeSolution(String fileName) throws IOException {
        try (PrintWriter output = new PrintWriter(fileName)) {
            output.println(surface.getDoubleValue() + " " + volume.getDoubleValue());
            output.println(R.getDoubleValue() + " " + r.getDoubleValue() + " " + h.getDoubleValue());
        }
    }

    public static void main(String[] args) {

        String outputFile = args.length > 0 ? args[0] : null;
        String strTimeLimit = args.length > 1 ? args[1] : "2";

        try (HexalyOptimizer optimizer = new HexalyOptimizer()) {
            OptimalBucket model = new OptimalBucket(optimizer);
            model.solve(Integer.parseInt(strTimeLimit));
            if (outputFile != null) {
                model.writeSolution(outputFile);
            }
        } catch (Exception ex) {
            System.err.println(ex);
            ex.printStackTrace();
            System.exit(1);
        }
    }
}
