
using System;
using System.IO;
using System.Linq;
using Hexaly.Optimizer;

public class ParallelScheduling : IDisposable
{
    // Number of tasks
    private int nbTasks;

    // Number of machines
    private int nbMachines;

    // Lengths of the tasks
    private long[] taskLengths;

    // Lowerbound on the makespan
    private long makespanLB;

    // Set of tasks assigned to each machine
    private HxExpression[] machineTasks;

    // Makespan of each machine 
    private HxExpression[] machineMakespan;

    // Hexaly Optimizer
    private HexalyOptimizer optimizer;

    public ParallelScheduling() {
        optimizer = new HexalyOptimizer();
    }

    /* Read instance data */
    public void ReadInstance(string fileName)
    {
        using (StreamReader input = new StreamReader(fileName)){
            string[] line = input.ReadLine().Split(" ");
            nbTasks = int.Parse(line[2]);
            nbMachines = int.Parse(line[3]);
            taskLengths = new long[nbTasks];
            line = input.ReadLine().Split(" ");

            long totalLength = 0;
            for (int i = 0; i < nbTasks; i++){
                taskLengths[i] = int.Parse(line[i]);
                totalLength += taskLengths[i];
            }
            makespanLB = totalLength / nbMachines;
        }
    }

    public void Dispose()
    {
        if (optimizer != null)
            optimizer.Dispose();
    }

    public void Solve(int limit)
    {   
        // Declare the optimization model
        HxModel model = optimizer.GetModel();

        machineTasks = new HxExpression[nbMachines];
        machineMakespan = new HxExpression[nbMachines];

        //Set decisions: machineTasks[k] represents the tasks assigned to machine k
        for (int i = 0; i < nbMachines; i++)
            machineTasks[i] = model.Set(nbTasks);
        
        // Each task must be scheduled on exactly one machine
        model.AddConstraint(model.Partition(machineTasks));

        // Create an array and a lambda function to retrieve the tasks' lengths
        HxExpression lengths = model.Array(taskLengths);
        var lengthLambda = model.CreateLambdaFunction(iExpr => lengths[iExpr]);

        for (int i = 0; i < nbMachines; i++)
            machineMakespan[i] = model.Sum(machineTasks[i], lengthLambda);
        
        // Minimize the makespan
        HxExpression makespan = model.Max(machineMakespan);
        model.Minimize(makespan);
        
        model.Close();

        // Parametrize the optimizer
        optimizer.GetParam().SetTimeLimit(limit);

        // Stop the search if the lower threshold is reached
        optimizer.GetParam().SetObjectiveThreshold(0, makespanLB);

        optimizer.Solve();
    }

    /* Write the solution in a file */
    void WriteSolution(string fileName)
    {
        using (StreamWriter output = new StreamWriter(fileName))
        {
            for (int k = 0; k < nbMachines; ++k)
            {
                output.Write("Makespan machine " + k + ": "
                        + machineMakespan[k].GetValue() + " | Items: ");
                HxCollection machineCollection = machineTasks[k].GetCollectionValue();
                if (machineCollection.Count() == 0)
                    continue;
                for (int i = 0; i < machineCollection.Count(); ++i)
                    output.Write(machineCollection[i] + " ");
                output.WriteLine();
            }
        }
    }
    
    public static void Main(string[] args)
    {
        if (args.Length < 1)
        {
            Console.Error.WriteLine("Usage: ParallelScheduling inputFile [outputFile] [timeLimit]");
            return;
        }
        string instanceFile = args[0];
        string solFile = args.Length > 1 ? args[1] : null;
        string strTimeLimit = args.Length > 2 ? args[2] : "5";
        try
        {
            var model = new ParallelScheduling();
            model.ReadInstance(instanceFile);
            if (!int.TryParse(strTimeLimit, out int limit))
                limit = 5;
            model.Solve(limit);
            if (solFile != null)
                model.WriteSolution(solFile);
        }
        catch (Exception e)
        {
            Console.Error.WriteLine("An error occurred: " + e.Message);
        }
    }
}
