import java.util.*;
import java.io.*;
import com.hexaly.optimizer.*;

public class ParallelScheduling {
    // Number of tasks
    private int nbTasks;

    // Number of machines
    private int nbMachines;

    // Lengths of the tasks
    private long[] taskLengths;

    // Lowerbound on the makespan
    private long makespanLB;

    // Set of tasks assigned to machine
    private HxExpression[] machineTasks;

    // Makespan of each machine 
    private HxExpression[] machineMakespan;

    // Hexaly Optimizer
    private final HexalyOptimizer optimizer;

    private ParallelScheduling(HexalyOptimizer optimizer) {
        this.optimizer = optimizer;
    }

    /* Read instance data */
    private void readInstance(String fileName) throws IOException {
        try (Scanner input = new Scanner(new File(fileName))) {
            input.useLocale(Locale.ROOT);
            String[] line = input.nextLine().split(" ");
            nbTasks = Integer.parseInt(line[2]);
            nbMachines = Integer.parseInt(line[3]);

            taskLengths = new long[nbTasks];
            long totalLengths = 0;
            for (int i = 0; i < nbTasks; ++i) {
                taskLengths[i] = input.nextInt();
                totalLengths += taskLengths[i];
            }
            makespanLB = totalLengths / nbMachines;
        }
    }

    private void solve(int limit) {
        // Declare the optimization model
        HxModel model = optimizer.getModel();

        machineTasks = new HxExpression[nbMachines];
        machineMakespan = new HxExpression[nbMachines];

        // Set decisions: machineTasks[k] represents the tasks in machine k
        for (int k = 0; k < nbMachines; ++k) 
            machineTasks[k] = model.setVar(nbTasks);

        // Each task must be scheduled on exactly one machine
        model.constraint(model.partition(machineTasks));

        // Create an array and a lambda function to retrieve the tasks' lengths
        HxExpression lengths = model.array(taskLengths);
        HxExpression lengthLambda = model.lambdaFunction(i -> model.at(lengths, i));

        for (int k = 0; k < nbMachines; ++k)
            machineMakespan[k] = model.sum(machineTasks[k], lengthLambda);

        // Minimize the makespan
        HxExpression makespan = model.max(machineMakespan);
        model.minimize(makespan);
        
        model.close();

        // Parametrize the optimizer
        optimizer.getParam().setTimeLimit(limit);

        // Stop the search if the lower threshold is reached
        optimizer.getParam().setObjectiveThreshold(0, makespanLB);

        optimizer.solve();
    }

    /* Write the solution in a file */
    private void writeSolution(String fileName) throws IOException {
        try (PrintWriter output = new PrintWriter(fileName)) {
            for (int k = 0; k < nbMachines; ++k) {
                output.print("Makespan machine " + k + ": " 
                        + machineMakespan[k].getValue() + " | Items: ");
                HxCollection machineCollection = machineTasks[k].getCollectionValue();
                if (machineCollection.count() == 0)
                    continue;
                for (int i = 0; i < machineCollection.count(); ++i) 
                    output.print(machineCollection.get(i) + " ");
                output.println();
            }
        }
    }


    public static void main(String[] args) {
        if (args.length < 1) {
            System.err.println("Usage: java ParallelScheduling inputFile [outputFile] [timeLimit]");
            System.exit(1);
        }
        String instanceFile = args[0];
        String outputFile = args.length > 1 ? args[1] : null;
        String strTimeLimit = args.length > 2 ? args[2] : "5";

        try (HexalyOptimizer optimizer = new HexalyOptimizer()) {
            ParallelScheduling model = new ParallelScheduling(optimizer);
            model.readInstance(instanceFile);
            model.solve(Integer.parseInt(strTimeLimit));
            if (outputFile != null) {
                model.writeSolution(outputFile);
            }
        } catch (Exception ex) {
            System.err.println(ex);
            ex.printStackTrace();
            System.exit(1);
        }
    }
}
