#include "optimizer/hexalyoptimizer.h"
#include <cmath>
#include <fstream>
#include <iostream>
#include <numeric>
#include <vector>

using namespace hexaly; 
using namespace std;

class ParallelScheduling {
private:
    // Number of tasks
    int nbTasks;

    // Number of machines
    int nbMachines;

    // Lengths of the tasks
    vector<int> taskLengths;

    // Lowerbound on the makespan
    int makespanLB;

    // Set of tasks assigned to machine
    vector<HxExpression> machineTasks;

    // Makespan of each machine
    vector<HxExpression> machineMakespan;

    // Hexaly Optimizer
    HexalyOptimizer optimizer;
    
public:
    /* Read instance data */
    void readInstance(const string& fileName) {
        ifstream infile;
        infile.exceptions(ifstream::failbit | ifstream::badbit);
        infile.open(fileName.c_str());

        string dump;
        infile >> dump;
        infile >> dump;
        infile >> nbTasks;
        infile >> nbMachines;

        taskLengths.resize(nbTasks);
        int totalLength = 0;
        for (int i = 0; i < nbTasks; ++i) {
            infile >> taskLengths[i];
            totalLength += taskLengths[i];
        }
        makespanLB = totalLength / nbMachines;
    }

    void solve(int limit) {
        // Declare the optimization model
        HxModel model = optimizer.getModel();

        machineTasks.resize(nbMachines);
        machineMakespan.resize(nbMachines);
        
        // Set decisions: machineTasks[k] represents the tasks assigned to machine k
        for (int i = 0; i < nbMachines; i++){
            machineTasks[i] = model.setVar(nbTasks);
        }

        // Each task must be scheduled on exactly one machine
        model.addConstraint(model.partition(machineTasks.begin(), machineTasks.end()));

        // Create an array and a lambda function to retrieve the tasks' lengths
        HxExpression lengths = model.array(taskLengths.begin(), taskLengths.end());
        HxExpression lengthLambda = model.createLambdaFunction(
                [&](HxExpression i){ return lengths[i]; });

        for (int i = 0; i < nbMachines; i++){
            machineMakespan[i] = model.sum(machineTasks[i], lengthLambda);
        }
        // Minimize the makespan
        HxExpression makespan = model.max(machineMakespan.begin(),
                machineMakespan.end());
        model.minimize(makespan);

        model.close();

        // Parametrize the solver
        optimizer.getParam().setTimeLimit(limit);

        // Stop the search if the lower threshold is reached
        optimizer.getParam().setObjectiveThreshold(0, (hxint) makespanLB);

        optimizer.solve();
    }
    
    /* Write the solution in a file */
    void writeSolution(const string& fileName) {
        ofstream outfile;
        outfile.exceptions(ofstream::failbit | ofstream::badbit);
        outfile.open(fileName.c_str());
        for (int k = 0; k < nbMachines; ++k) {
            outfile << "Makespan machine " << k << ": "
                    << machineMakespan[k].getValue() << " | Items: ";
            HxCollection machineCollection = machineTasks[k].getCollectionValue();
            if (machineCollection.count() == 0)
                continue;
            for (int i = 0; i < machineCollection.count(); ++i) {
                outfile << machineCollection[i] << " ";
            }
            outfile << endl;
        }
    }
};

int main(int argc, char** argv) {
    if (argc < 2) {
        cerr << "Usage: parallel_scheduling inputFile [outputFile] [timeLimit]" << endl;
        return 1;
    }
    const char* instanceFile = argv[1];
    const char* solFile = argc > 2 ? argv[2] : NULL;
    const char* strTimeLimit = argc > 3 ? argv[3] : "5";
    try {
        ParallelScheduling model;
        model.readInstance(instanceFile);
        model.solve(atoi(strTimeLimit));
        if (solFile != NULL)
            model.writeSolution(solFile);
        return 0;
    } catch (const exception& e) {
        cerr << "An error occurred: " << e.what() << endl;
        return 1;
    }
}
