import hexaly.optimizer
import sys
import math

if len(sys.argv) < 2:
    print("Usage: python parallel_scheduling.py inputFile [outputFile] [timeLimit]")
    sys.exit(1)

with hexaly.optimizer.HexalyOptimizer() as optimizer:
    # Read instance data
    filename = sys.argv[1]

    with open(filename) as f:
        line = f.readline()
        _, _, n_tasks, n_machines = [elem for elem in line.split()]

        # Number of tasks
        n_tasks = int(n_tasks)

        # Number of machines
        n_machines = int(n_machines)

        # Lengths of the tasks
        task_lengths = [int(elem) for elem in f.readline().split() if elem != '0']

        # Lowerbound on the makespan
        makespan_lb = int(sum(task_lengths) / n_machines) 

    # Declare the optimization model
    model = optimizer.model

    # Set decisions: machineTasks[k] represents the tasks assigned to machine k
    machine_tasks = [model.set(n_tasks) for _ in range(n_machines)]

    # Each task must be scheduled on exactly one machine
    model.constraint(model.partition(machine_tasks))

    # Create an array and a lambda function to retrieve the tasks' lengths
    lengths = model.array(task_lengths)
    lengths_lambda = model.lambda_function(lambda i: lengths[i])

    # Minimize the makespan
    machine_makespan = [model.sum(i, lengths_lambda) for i in machine_tasks]
    makespan = model.max(machine_makespan)
    model.minimize(makespan)

    model.close()

    # Parametrize the optimizer
    if len(sys.argv) >= 4:
        optimizer.param.time_limit = int(sys.argv[3])
    else:
        optimizer.param.time_limit = 5

    # Stop the search if the lower threshold is reached
    optimizer.param.set_objective_threshold(0, makespan_lb)

    optimizer.solve()

    # Write the solution in a file
    if len(sys.argv) >= 3:
        with open(sys.argv[2], 'w') as f:
            for k in range(n_machines):
                f.write("Makespan machine {}: {} | Items: ".format(k, machine_makespan[k].value))
                if len(machine_tasks[k].value) == 0:
                    continue
                for e in machine_tasks[k].value:
                    f.write("%d " % e)
                f.write("\n")
