import java.util.*;
import java.io.*;
import com.hexaly.optimizer.*;

public class Pcvrp {
    // Hexaly Optimizer
    private final HexalyOptimizer optimizer;

    // Number of customers
    int nbCustomers;

    // Capacity of the trucks
    private int truckCapacity;

    // Minimum number of demands to satisfy
    private long demandsToSatisfy;

    // Demand on each customer
    private long[] demandsData;

    // Prize on each customer
    private long[] prizesData;

    // Distance matrix between customers
    private long[][] distMatrixData;

    // Distances between customers and depot
    private long[] distDepotData;

    // Number of trucks
    private int nbTrucks;

    // Decision variables
    private HxExpression[] customersSequences;

    // Are the trucks actually used
    private HxExpression[] trucksUsed;

    // Number of trucks used in the solution
    private HxExpression nbTrucksUsed;

    // Distance traveled by all the trucks
    private HxExpression totalDistance;

    // Total prize of the solution
    private HxExpression totalPrize;

    // Total nb demands satisfied in the solution
    private HxExpression totalQuantity;

    private Pcvrp(HexalyOptimizer optimizer) {
        this.optimizer = optimizer;
    }

    private void solve(int limit) {
        // Declare the optimization model
        HxModel model = optimizer.getModel();

        trucksUsed = new HxExpression[nbTrucks];
        customersSequences = new HxExpression[nbTrucks];
        HxExpression[] distRoutes = new HxExpression[nbTrucks];
        HxExpression[] routeQuantities = new HxExpression[nbTrucks];
        HxExpression[] routePrizes = new HxExpression[nbTrucks];

        // Sequence of customers visited by each truck.
        for (int k = 0; k < nbTrucks; ++k)
            customersSequences[k] = model.listVar(nbCustomers);

        // A customer might be visited by only one truck
        model.constraint(model.disjoint(customersSequences));

        // Create HexalyOptimizer arrays to be able to access them with an "at" operator
        HxExpression demands = model.array(demandsData);
        HxExpression prizes = model.array(prizesData);
        HxExpression distDepot = model.array(distDepotData);
        HxExpression distMatrix = model.array(distMatrixData);

        for (int k = 0; k < nbTrucks; ++k) {
            HxExpression sequence = customersSequences[k];
            HxExpression c = model.count(sequence);

            // A truck is used if it visits at least one customer
            trucksUsed[k] = model.gt(c, 0);

            // The quantity needed in each route must not exceed the truck capacity
            HxExpression demandLambda = model.lambdaFunction(j -> model.at(demands, j));
            routeQuantities[k] = model.sum(sequence, demandLambda);
            model.constraint(model.leq(routeQuantities[k], truckCapacity));

            // Distance traveled by truck k
            HxExpression distLambda = model
                .lambdaFunction(i -> model.at(distMatrix, model.at(sequence, model.sub(i, 1)), model.at(sequence, i)));
            distRoutes[k] = model.sum(model.sum(model.range(1, c), distLambda),
                model.iif(model.gt(c, 0), model.sum(model.at(distDepot, model.at(sequence, 0)),
                    model.at(distDepot, model.at(sequence, model.sub(c, 1)))), 0));
            
            // Route prize of truck k
            HxExpression prizeLambda = model.lambdaFunction(j -> model.at(prizes, j));
            routePrizes[k] = model.sum(sequence, prizeLambda);
        }
        
        // Minimal number of demands to satisfy
        totalQuantity = model.sum(routeQuantities);
        model.constraint(model.geq(totalQuantity, demandsToSatisfy));

        totalPrize = model.sum(routePrizes);
        nbTrucksUsed = model.sum(trucksUsed);
        totalDistance = model.sum(distRoutes);

        // Objective: minimize the number of trucks used, then maximize the total prize and minimize the distance traveled
        model.minimize(nbTrucksUsed);
        model.maximize(totalPrize);
        model.minimize(totalDistance);

        model.close();

        // Parametrize the optimizer
        optimizer.getParam().setTimeLimit(limit);

        optimizer.solve();
    }

    /* Write the solution in a file with the following format:
    * - total prize, number of trucks used and total distance
    * - for each truck the customers visited (omitting the start/end at the depot)
    * - number of unvisited customers, demands satisfied */
    private void writeSolution(String fileName) throws IOException {
        try (PrintWriter output = new PrintWriter(fileName)) {
            output.println(totalPrize.getValue() + " " + nbTrucksUsed.getValue() + " " + totalDistance.getValue());
            int nbUnvisitedCustomers = nbCustomers;
            for (int k = 0; k < nbTrucks; ++k) {
                if (trucksUsed[k].getValue() != 1)
                    continue;
                // Values in sequence are in 0...nbCustomers. +1 is to put it back in 1...nbCustomers+1
                // as in the data files (0 being the depot)
                HxCollection customersCollection = customersSequences[k].getCollectionValue();
                for (int i = 0; i < customersCollection.count(); ++i) {
                    output.print((customersCollection.get(i) + 1) + " ");
                    --nbUnvisitedCustomers;
                }
                output.println();
            }
            output.println(nbUnvisitedCustomers + " " + totalQuantity.getValue());
        }
    }

    // The input files follow the "longjianyu" format
    private void readInstance(String fileName) throws IOException {
        try (Scanner input = new Scanner(new File(fileName))) {
            input.useLocale(Locale.ROOT);
            nbTrucks = input.nextInt();
            truckCapacity = input.nextInt();
            demandsToSatisfy = input.nextInt();

            int n = 0;
            ArrayList<Integer> customersXList = new ArrayList<>(), customersYList = new ArrayList<>();
            ArrayList<Long> demandsDataList = new ArrayList<>(), prizesDataList = new ArrayList<>();
            int depotX = 0, depotY = 0;
            while (input.hasNext()) {
                int id = input.nextInt();
                if (id != n)
                    throw new IOException("Unexpected index");
                if (n == 0) {
                    depotX = input.nextInt();
                    depotY = input.nextInt();
                } else {
                    customersXList.add(input.nextInt());
                    customersYList.add(input.nextInt());
                    demandsDataList.add(input.nextLong());
                    prizesDataList.add(input.nextLong());
                }
                ++n;
            }
            nbCustomers = n - 1;

            int[] customersX = customersXList.stream().mapToInt(i -> i).toArray();
            int[] customersY = customersYList.stream().mapToInt(i -> i).toArray();
            
            demandsData = demandsDataList.stream().mapToLong(i -> i).toArray();
            prizesData = prizesDataList.stream().mapToLong(i -> i).toArray();

            computeDistanceMatrix(depotX, depotY, customersX, customersY);
        }
    }

    // Compute the distance matrix
    private void computeDistanceMatrix(int depotX, int depotY, int[] customersX, int[] customersY) {
        distMatrixData = new long[nbCustomers][nbCustomers];
        for (int i = 0; i < nbCustomers; ++i) {
            distMatrixData[i][i] = 0;
            for (int j = i + 1; j < nbCustomers; ++j) {
                long dist = computeDist(customersX[i], customersX[j], customersY[i], customersY[j]);
                distMatrixData[i][j] = dist;
                distMatrixData[j][i] = dist;
            }
        }

        distDepotData = new long[nbCustomers];
        for (int i = 0; i < nbCustomers; ++i) {
            distDepotData[i] = computeDist(depotX, customersX[i], depotY, customersY[i]);
        }
    }

    private long computeDist(int xi, int xj, int yi, int yj) {
        double exactDist = Math.sqrt(Math.pow(xi - xj, 2) + Math.pow(yi - yj, 2));
        return Math.round(exactDist);
    }

    public static void main(String[] args) {
        if (args.length < 1) {
            System.err.println("Usage: java Pcvrp inputFile [outputFile] [timeLimit]");
            System.exit(1);
        }

        try (HexalyOptimizer optimizer = new HexalyOptimizer()) {
            String instanceFile = args[0];
            String outputFile = args.length > 1 ? args[1] : null;
            String strTimeLimit = args.length > 2 ? args[2] : "20";

            Pcvrp model = new Pcvrp(optimizer);
            model.readInstance(instanceFile);
            model.solve(Integer.parseInt(strTimeLimit));
            if (outputFile != null) {
                model.writeSolution(outputFile);
            }
        } catch (Exception ex) {
            System.err.println(ex);
            ex.printStackTrace();
            System.exit(1);
        }
    }
}
