import hexaly.optimizer
import sys
import math


def read_elem(filename):
    with open(filename) as f:
        return [str(elem) for elem in f.read().split()]


def main(instance_file, str_time_limit, output_file):
    #
    # Read instance data
    #
    nb_customers, nb_trucks, truck_capacity, dist_matrix_data, dist_depot_data, \
        demands_data, demands_to_satisfy, prizes_data = read_input_pcvrp(instance_file)

    with hexaly.optimizer.HexalyOptimizer() as optimizer:
        #
        # Declare the optimization model
        #
        model = optimizer.model

        # Sequence of customers visited by each truck
        customers_sequences = [model.list(nb_customers) for _ in range(nb_trucks)]

        # A customer might be visited by only one truck
        model.constraint(model.disjoint(customers_sequences))

        # Create Hexaly arrays to be able to access them with an "at" operator
        demands = model.array(demands_data)
        prizes = model.array(prizes_data)
        dist_matrix = model.array(dist_matrix_data)
        dist_depot = model.array(dist_depot_data)

        # A truck is used if it visits at least one customer
        trucks_used = [(model.count(customers_sequences[k]) > 0) for k in range(nb_trucks)]

        dist_routes = [None] * nb_trucks
        route_prizes = [None] * nb_trucks
        route_quantities = [None] * nb_trucks

        for k in range(nb_trucks):
            sequence = customers_sequences[k]
            c = model.count(sequence)

            # The quantity needed in each route must not exceed the truck capacity
            demand_lambda = model.lambda_function(lambda j: demands[j])
            route_quantities[k] = model.sum(sequence, demand_lambda)
            model.constraint(route_quantities[k] <= truck_capacity)

            # Distance traveled by each truck
            dist_lambda = model.lambda_function(lambda i:
                                                model.at(dist_matrix,
                                                         sequence[i - 1],
                                                         sequence[i]))
            dist_routes[k] = model.sum(model.range(1, c), dist_lambda) \
                + model.iif(c > 0,
                            dist_depot[sequence[0]] + dist_depot[sequence[c - 1]],
                            0)
            
            # Route prize of truck k
            prize_lambda = model.lambda_function(lambda j: prizes[j])
            route_prizes[k] = model.sum(sequence, prize_lambda)

        # Total nb demands satisfied
        total_quantity = model.sum(route_quantities)
    
        # Minimal number of demands to satisfy
        model.constraint(total_quantity >= demands_to_satisfy)

        # Total nb trucks used
        nb_trucks_used = model.sum(trucks_used)

        # Total distance traveled
        total_distance = model.sum(dist_routes)

        # Total prize
        total_prize = model.sum(route_prizes)

        # Objective: minimize the number of trucks used, then maximize the total prize and minimize the distance traveled
        model.minimize(nb_trucks_used)
        model.maximize(total_prize)
        model.minimize(total_distance)

        model.close()

        # Parameterize the optimizer
        optimizer.param.time_limit = int(str_time_limit)

        optimizer.solve()

        #
        # Write the solution in a file with the following format:
        #  - total prize, number of trucks used and total distance
        #  - for each truck the customers visited (omitting the start/end at the depot)
        #  - number of unvisited customers, demands satisfied
        if output_file is not None:
            with open(output_file, 'w') as f:
                f.write("%d %d %d\n" % (total_prize.value, nb_trucks_used.value, total_distance.value))
                nb_unvisited_customers = nb_customers
                for k in range(nb_trucks):
                    if trucks_used[k].value != 1:
                        continue
                    # Values in sequence are in 0...nbCustomers. +1 is to put it back in 1...nbCustomers+1
                    # as in the data files (0 being the depot)
                    for customer in customers_sequences[k].value:
                        f.write("%d " % (customer + 1))
                        nb_unvisited_customers -= 1
                    f.write("\n")
                f.write("%d %d\n" % (nb_unvisited_customers, total_quantity.value))


# The input files follow the "longjianyu" format
def read_input_pcvrp(filename):
    file_it = iter(read_elem(filename))

    nb_trucks = int(next(file_it))
    truck_capacity = int(next(file_it))
    demands_to_satisfy = int(next(file_it))

    n = 0
    customers_x = []
    customers_y = []
    depot_x = 0
    depot_y = 0
    demands = []
    prizes = []
    
    it = next(file_it, None)
    while (it != None):
        node_id = int(it)
        if node_id != n:
            print("Unexpected index")
            sys.exit(1)

        if n == 0:
            depot_x = int(next(file_it))
            depot_y = int(next(file_it))
        else:
            customers_x.append(int(next(file_it)))
            customers_y.append(int(next(file_it)))
            demands.append(int(next(file_it)))
            prizes.append(int(next(file_it)))
        it = next(file_it, None)
        n += 1

    distance_matrix = compute_distance_matrix(customers_x, customers_y)
    distance_depots = compute_distance_depots(depot_x, depot_y, customers_x, customers_y)

    nb_customers = n - 1

    return nb_customers, nb_trucks, truck_capacity, distance_matrix, distance_depots, \
        demands, demands_to_satisfy, prizes


# Compute the distance matrix
def compute_distance_matrix(customers_x, customers_y):
    nb_customers = len(customers_x)
    distance_matrix = [[None for _ in range(nb_customers)] for _ in range(nb_customers)]
    for i in range(nb_customers):
        distance_matrix[i][i] = 0
        for j in range(nb_customers):
            dist = compute_dist(customers_x[i], customers_x[j], customers_y[i], customers_y[j])
            distance_matrix[i][j] = dist
            distance_matrix[j][i] = dist
    return distance_matrix


# Compute the distances to depot
def compute_distance_depots(depot_x, depot_y, customers_x, customers_y):
    nb_customers = len(customers_x)
    distance_depots = [None] * nb_customers
    for i in range(nb_customers):
        dist = compute_dist(depot_x, customers_x[i], depot_y, customers_y[i])
        distance_depots[i] = dist
    return distance_depots


def compute_dist(xi, xj, yi, yj):
    exact_dist = math.sqrt(math.pow(xi - xj, 2) + math.pow(yi - yj, 2))
    return int(math.floor(exact_dist + 0.5))


if __name__ == '__main__':
    if len(sys.argv) < 2:
        print("Usage: python pcvrp.py input_file [output_file] [time_limit]")
        sys.exit(1)

    instance_file = sys.argv[1]
    output_file = sys.argv[2] if len(sys.argv) > 2 else None
    str_time_limit = sys.argv[3] if len(sys.argv) > 3 else "20"

    main(instance_file, str_time_limit, output_file)
