using System;
using System.IO;
using System.Linq;
using System.Globalization;
using Hexaly.Optimizer;

public class Portfolio : IDisposable
{
    // Expected profit, in percentage of the portfolio
    private float expectedProfit;

    // Number of stocks
    private int nbStocks;

    // Covariance among the stocks
    private float[][] sigmaStocks;

    // Variation of the price of each stock
    private float[] deltaStock;

    // Hexaly Optimizer
    private HexalyOptimizer optimizer;

    // Proportion of the portfolio invested in each stock
    private HxExpression[] portfolioStock;

    // Return of the portfolio in percentage
    private HxExpression profit;

    public Portfolio()
    {
        optimizer = new HexalyOptimizer();
    }

    public void ReadInstance(string fileName)
    {
        using (StreamReader input = new StreamReader(fileName))
        {
            char[] separators = new char[] { '\t', ' ' };
            string[] splitted = input
                .ReadLine()
                .Split(separators, StringSplitOptions.RemoveEmptyEntries);
            expectedProfit = float.Parse(splitted[0], CultureInfo.InvariantCulture);
    
            input.ReadLine();
            splitted = input
                .ReadLine()
                .Split(separators, StringSplitOptions.RemoveEmptyEntries);
            nbStocks = int.Parse(splitted[0]);

            sigmaStocks = new float[nbStocks][];
            input.ReadLine();
            for (int s = 0; s < nbStocks; ++s)
            {
                sigmaStocks[s] = new float[nbStocks];
                splitted = input
                    .ReadLine()
                    .Split(separators, StringSplitOptions.RemoveEmptyEntries);
                for (int t = 0; t < nbStocks; ++t)
                    sigmaStocks[s][t] = float.Parse(splitted[t], CultureInfo.InvariantCulture);
            }

            input.ReadLine();
            splitted = input
                .ReadLine()
                .Split(separators, StringSplitOptions.RemoveEmptyEntries);

            deltaStock = new float[nbStocks];
            for(int s = 0; s < nbStocks; ++s)
                deltaStock[s] = float.Parse(splitted[s], CultureInfo.InvariantCulture);
        }
    }

    public void Dispose()
    {
        optimizer.Dispose();
    }

    public void Solve(int timeLimit)
    {
        // Declare the optimization model
        HxModel model = optimizer.GetModel();

        // Proportion of the portfolio invested in each stock
        portfolioStock = new HxExpression[nbStocks];
        for(int s = 0; s < nbStocks; s++)
            portfolioStock[s] = model.Float(0.0, 1.0);

        // Risk of the portfolio
        HxExpression risk = model.Sum();
        for(int s = 0; s < nbStocks; s++)
        {
            for(int t = 0; t < nbStocks; t++)
                risk.AddOperand(portfolioStock[s] * portfolioStock[t] * sigmaStocks[s][t]);
        }

        // Return of the portfolio in percentage
        profit = model.Sum();
        for(int s = 0; s < nbStocks; s++)
            profit.AddOperand(portfolioStock[s] * deltaStock[s]);

        // All the portfolio is used
        model.Constraint(model.Sum(portfolioStock) == 1.0);

        // The profit is at least the expected profit
        model.Constraint(profit >= expectedProfit);

        // Minimize the risk
        model.Minimize(risk);

        model.Close();

        // Parameterize the optimizer
        optimizer.GetParam().SetTimeLimit(timeLimit);

        optimizer.Solve();
    }

    /* Write the solution in a file with the following format:
     *  - for each stock, the proportion of the porfolio invested
        - the final profit in percentage of the portfolio */
    public void WriteSolution(string fileName)
    {
        using (StreamWriter output = new StreamWriter(fileName))
        {
            Console.WriteLine("Solution written in file " + fileName);
            for (int s = 1; s <= nbStocks; ++s)
            {
                double proportion = portfolioStock[s - 1].GetDoubleValue();
                output.WriteLine("Stock " + s + ": " +
                    Math.Round(proportion * 100, 1) + "%");
            }
            output.WriteLine("Profit: " + Math.Round(profit.GetDoubleValue(), 4) + "%");
        }
    }

    public static void Main(string[] args)
    {
        if (args.Length < 1)
        {
            Console.WriteLine("Usage: Portfolio instanceFile [outputFile] [timeLimit]");
            System.Environment.Exit(1);
        }

        string instanceFile = args[0];
        string outputFile = args.Length > 1 ? args[1] : null;
        string strTimeLimit = args.Length > 2 ? args[2] : "60";

        using (Portfolio model = new Portfolio())
        {
            model.ReadInstance(instanceFile);
            model.Solve(int.Parse(strTimeLimit));
            if (outputFile != null)
                model.WriteSolution(outputFile);
        }
    }
}
