import java.io.File;
import java.io.IOException;
import java.io.PrintWriter;
import java.util.*;

import com.hexaly.optimizer.HxExpression;
import com.hexaly.optimizer.HxModel;
import com.hexaly.optimizer.HexalyOptimizer;

public class Portfolio {
    // Expected profit, in percentage of the portfolio
    private float expectedProfit;
    // Number of stocks
    private int nbStocks;
    // Covariance among the stocks
    private float[][] sigmaStocks;
    // Variation of the price of each stock
    private float[] deltaStock;

    // Hexaly Optimizer
    private final HexalyOptimizer optimizer;
    // Proportion of the portfolio invested in each stock
    private HxExpression[] portfolioStock;
    // Return of the portfolio in percentage
    private HxExpression profit;

    public Portfolio(HexalyOptimizer optimizer) throws IOException {
        this.optimizer = optimizer;
    }

    public void readInstance(String fileName) throws IOException {
        try (Scanner input = new Scanner(new File(fileName))) {
            input.useLocale(Locale.ROOT);
            expectedProfit = input.nextFloat();
            nbStocks = input.nextInt();

            sigmaStocks = new float[nbStocks][];
            for (int s = 0; s < nbStocks; ++s) {
                sigmaStocks[s] = new float[nbStocks];
                for (int t = 0; t < nbStocks; ++t) {
                    sigmaStocks[s][t] = input.nextFloat();
                }
            }

            deltaStock = new float[nbStocks];
            for (int s = 0; s < nbStocks; ++s) {
                deltaStock[s] = input.nextFloat();
            }
        }

    }

    public void solve(int timeLimit) {
        // Declare the optimization model
        HxModel model = optimizer.getModel();

        // Proportion of the portfolio invested in each stock
        portfolioStock = new HxExpression[nbStocks];
        for (int s = 0; s < nbStocks; s++) {
            portfolioStock[s] = model.floatVar(0.0, 1.0);
        }

        // Risk of the portfolio
        HxExpression risk = model.sum();
        for (int s = 0; s < nbStocks; s++) {
            for (int t = 0; t < nbStocks; t++) {
                HxExpression sigmaST = model.createConstant(sigmaStocks[s][t]);
                risk.addOperand(model.prod(portfolioStock[s], portfolioStock[t], sigmaST));
            }
        }

        // Return of the portfolio in percentage
        profit = model.sum();
        for (int s = 0; s < nbStocks; s++) {
            HxExpression deltaS = model.createConstant(deltaStock[s]);
            profit.addOperand(model.prod(portfolioStock[s], deltaS));
        }

        // All the portfolio is used
        HxExpression one = model.createConstant(1.0);
        model.constraint(model.eq(model.sum(portfolioStock), one));

        // The profit is at least the expected profit
        model.constraint(model.geq(profit, expectedProfit));

        // Minimize the risk
        model.minimize(risk);

        model.close();

        // Parameterize the optimizer
        optimizer.getParam().setTimeLimit(timeLimit);

        optimizer.solve();
    }

    /*
     * Write the solution in a file with the following format:
     * - for each stock, the proportion of the porfolio invested
     * - the final profit in percentage of the portfolio
     */
    public void writeSolution(String fileName) throws IOException {
        try (PrintWriter output = new PrintWriter(fileName)) {
            System.out.println("Solution written in file " + fileName);
            for (int s = 1; s <= nbStocks; ++s) {
                output.write("Stock " + s + ": " + portfolioStock[s - 1].getDoubleValue() * 100 + "% \n");
            }
            output.write("Profit: " + profit.getDoubleValue() + "% \n");
        }
    }

    public static void main(String[] args) {
        if (args.length < 1) {
            System.out.println("Usage: java Portfolio instanceFile [outputFile] [timeLimit]");
            System.exit(1);
        }

        String instanceFile = args[0];
        String outputFile = args.length > 1 ? args[1] : null;
        String strTimeLimit = args.length > 2 ? args[2] : "60";

        try (HexalyOptimizer optimizer = new HexalyOptimizer()) {
            Portfolio model = new Portfolio(optimizer);
            model.readInstance(instanceFile);
            model.solve(Integer.parseInt(strTimeLimit));
            if (outputFile != null) {
                model.writeSolution(outputFile);
            }
        } catch (Exception ex) {
            System.err.println(ex);
            ex.printStackTrace();
            System.exit(1);
        }
    }
}
