#include "optimizer/hexalyoptimizer.h"
#include <algorithm>
#include <fstream>
#include <iostream>
#include <limits>
#include <numeric>
#include <vector>

using namespace hexaly;

class Portfolio {
private:
    // Expected profit, in percentage of the portfolio
    float expectedProfit;
    // Number of stocks
    int nbStocks;
    // Covariance among the stocks
    std::vector<std::vector<float>> sigmaStocks;
    // Variation of the price of each stock
    std::vector<float> deltaStock;

    // Hexaly Optimizer
    HexalyOptimizer optimizer;
    // Proportion of the portfolio invested in each stock
    std::vector<HxExpression> portfolioStock;
    // Return of the portfolio in percentage
    HxExpression profit;

public:
    Portfolio() : optimizer() {}

    void readInstance(const std::string &fileName) {
        std::ifstream infile;
        infile.exceptions(std::ifstream::failbit | std::ifstream::badbit);
        infile.open(fileName.c_str());

        infile >> expectedProfit;
        infile >> nbStocks;

        for (int s = 0; s < nbStocks; s++) {
            sigmaStocks.push_back(std::vector<float>(nbStocks, 0.0));
            for (int t = 0; t < nbStocks; t++) {
                infile >> sigmaStocks[s][t];
            }
        }

        deltaStock = std::vector<float>(nbStocks, 0.0);
        for (int s = 0; s < nbStocks; s++) {
            infile >> deltaStock[s];
            deltaStock[s] = deltaStock[s];
        }

        infile.close();
    }

    void solve(int timeLimit) {
        // Declare the optimization model
        HxModel model = optimizer.getModel();

        // Proportion of the portfolio invested in each stock
        portfolioStock.resize(nbStocks);
        for (int s = 0; s < nbStocks; s++)
            portfolioStock[s] = model.floatVar(0.0, 1.0);

        // Risk of the portfolio
        HxExpression risk = model.sum();
        for (int s = 0; s < nbStocks; s++) {
            for (int t = 0; t < nbStocks; t++) {
                risk.addOperand(portfolioStock[s] * portfolioStock[t] * sigmaStocks[s][t]);
            }
        }

        // Return of the portfolio in percentage
        profit = model.sum();
        for (int s = 0; s < nbStocks; s++) {
            profit.addOperand(portfolioStock[s] * deltaStock[s]);
        }

        // All the portfolio is used
        model.constraint(model.sum(portfolioStock.begin(), portfolioStock.end()) == 1.0);

        // The profit is at least the expected profit
        model.constraint(profit >= expectedProfit);

        // Minimize the risk
        model.minimize(risk);

        model.close();

        // Parameterize the optimizer
        optimizer.getParam().setTimeLimit(timeLimit);

        optimizer.solve();
    }

    /* Write the solution in a file with the following format:
        - for each stock, the proportion of the porfolio invested
        - the final profit in percentage of the portfolio */
    void writeSolution(const std::string &fileName) {
        std::ofstream outfile(fileName.c_str());
        if (!outfile.is_open()) {
            std::cerr << "File " << fileName << " cannot be opened." << std::endl;
            exit(1);
        }
        std::cout << "Solution written in file " << fileName << std::endl;

        for (unsigned int s = 0; s < nbStocks; ++s) {
            float proportion = portfolioStock[s].getDoubleValue();
            outfile << "Stock " << s + 1 << ": " << round(proportion * 1000) / 10 << "%" << std::endl;
        }
        outfile << "Profit: " << profit.getDoubleValue() << "%" << std::endl;
        outfile.close();
    }
};

int main(int argc, char **argv) {
    if (argc < 2) {
        std::cout << "Usage: portfolio instanceFile [outputFile] [timeLimit]" << std::endl;
        exit(1);
    }

    const char *instanceFile = argv[1];
    const char *outputFile = argc > 2 ? argv[2] : NULL;
    const char *strTimeLimit = argc > 3 ? argv[3] : "60";

    Portfolio model;
    try {
        model.readInstance(instanceFile);
        const int timeLimit = atoi(strTimeLimit);
        model.solve(timeLimit);
        if (outputFile != NULL)
            model.writeSolution(outputFile);
        return 0;
    } catch (const std::exception &e) {
        std::cerr << "An error occurred: " << e.what() << std::endl;
        return 1;
    }
}
