from re import S
import hexaly.optimizer
import sys


def read_instance(filename):
    with open(filename) as f:
        lines = f.readlines()

    first_line = lines[0].split()

    # Expected profit, in percentage of the portfolio
    expected_profit = float(first_line[0])

    second_line = lines[2].split()

    # Number of stocks
    nb_stocks = int(second_line[0])

    # Covariance among the stocks
    sigma_stocks = [[0 for i in range(nb_stocks)] for j in range(nb_stocks)]
    for s in range(nb_stocks):
        line = lines[s+4].split()
        for t in range(nb_stocks):
            sigma_stocks[s][t] = float(line[t])

    # Variation of the price of each stock
    delta_stock = [0 for i in range(nb_stocks)]
    line = lines[nb_stocks+5].split()
    for s in range(nb_stocks):
        delta_stock[s] = float(line[s])
        print(delta_stock[s])

    return expected_profit, nb_stocks, sigma_stocks, delta_stock


def main(instance_file, output_file, time_limit):
    expected_profit, nb_stocks, sigma_stocks, delta_stock = read_instance(
        instance_file)

    with hexaly.optimizer.HexalyOptimizer() as optimizer:
        #
        # Declare the optimization model
        #
        model = optimizer.model

        # Proportion of the portfolio invested in each stock
        portfolio_stock = [model.float(0, 1) for s in range(nb_stocks)]

        # Risk of the portfolio
        risk = model.sum(portfolio_stock[s] * portfolio_stock[t] * sigma_stocks[s][t]
                         for t in range(nb_stocks) for s in range(nb_stocks))

        # Return of the portfolio in percentage
        profit = model.sum(portfolio_stock[s] * delta_stock[s]
                           for s in range(nb_stocks))

        # All the portfolio is used
        model.constraint(
            model.sum(portfolio_stock[s] for s in range(nb_stocks)) == 1.0)

        # The profit is at least the expected profit
        model.constraint(profit >= expected_profit)

        # Minimize the risk
        model.minimize(risk)

        model.close()

        # Parameterize the optimizer
        optimizer.param.time_limit = time_limit

        optimizer.solve()

        # Write the solution in a file with the following format:
        # - for each stock, the proportion of the porfolio invested
        # - the final profit in percentage of the portfolio
        if output_file != None:
            with open(output_file, "w") as f:
                print("Solution written in file", output_file)
                for s in range(nb_stocks):
                    proportion = portfolio_stock[s].value
                    f.write("Stock " + str(s+1) + ": " + str(round(proportion * 100, 1))
                            + "%" + "\n")
                f.write("Profit: " + str(round(profit.value, 4)) + "%")


if __name__ == '__main__':
    if len(sys.argv) < 2:
        print(
            "Usage: python portfolio.py instance_file [output_file] [time_limit]")
        sys.exit(1)

    instance_file = sys.argv[1]
    output_file = sys.argv[2] if len(sys.argv) >= 3 else None
    time_limit = int(sys.argv[3]) if len(sys.argv) >= 4 else 60
    main(instance_file, output_file, time_limit)
