#include "optimizer/hexalyoptimizer.h"
#include <algorithm>
#include <fstream>
#include <iostream>
#include <limits>
#include <numeric>
#include <vector>

using namespace hexaly;

class Racp {
private:
    // Number of tasks
    int nbTasks;

    // Number of resources
    int nbResources;

    // Deadline of the project
    int deadline;

    // Cost per unit of capacity of each resource
    std::vector<int> resourceCost;

    // Duration of each task
    std::vector<int> duration;

    // Weight of resource r required for task i
    std::vector<std::vector<int>> weight; 

    // Number of successors
    std::vector<int> nbSuccessors;

    // Successors for each task i
    std::vector<std::vector<int>> successors;
    
    // Trivial upper bound for the end times of the tasks
    int horizon = 0;

    // Upper bound for the needed capacity of each resource
    std::vector<int> maxCapacity; 

    // Hexaly Optimizer
    HexalyOptimizer optimizer;

    // Decision variables: time range of each task
    std::vector<HxExpression> tasks;

    // Decision variables: capacity of each renewable resource
    std::vector<HxExpression> capacity;

    // Objective to minimize: end of the last task of the last job
    HxExpression makespan;
    
    // Objective to minimize: total cost of all resources
    HxExpression totalCost;

public:
    Racp() : optimizer() {}

    // The input files follow the "Patterson" format
    void readInstance(const std::string& fileName, const char* dline) {
        std::ifstream infile;
        infile.exceptions(std::ifstream::failbit | std::ifstream::badbit);
        infile.open(fileName.c_str());

        infile >> nbTasks;
        infile >> nbResources;

        resourceCost.resize(nbResources);
        for (int r = 0; r < nbResources; ++r) {
            infile >> resourceCost[r];
        }

        duration.resize(nbTasks);
        weight.resize(nbTasks);
        nbSuccessors.resize(nbTasks);
        successors.resize(nbTasks);

        for (int i = 0; i < nbTasks; ++i) {
            infile >> duration[i];  
            weight[i].resize(nbResources);
            for (int r = 0; r < nbResources; ++r) {
                infile >> weight[i][r];   
            }
            infile >> nbSuccessors[i];
            successors[i].resize(nbSuccessors[i]);
            for (int s = 0; s < nbSuccessors[i]; ++s) {
                int x;
                infile >> x;
                successors[i][s] = x - 1;
            }
            horizon += duration[i];
        }
            
        deadline = horizon;
        if (dline != NULL) {
            deadline = atoi(dline);
        }
        infile.close();
    }

    void solve(int timeLimit) {
        // Declare the optimization model
        HxModel model = optimizer.getModel();

        // Interval decision variables: time range of each task
        tasks.resize(nbTasks);
        for (int i = 0; i < nbTasks; ++i) {
            tasks[i] = model.intervalVar(0, horizon);

            // Task duration constraints
            model.constraint(model.length(tasks[i]) == duration[i]);
        }

        // Integer decision variables: capacity of each renewable resource
        capacity.resize(nbResources);
        maxCapacity.resize(nbResources);
        for (int r = 0; r < nbResources; ++r) {
            maxCapacity[r] = 0;
            for (int i = 0; i < nbTasks; ++i) {
                maxCapacity[r] += weight[i][r];
            }
            capacity[r] = model.intVar(0, maxCapacity[r]);
        }

        // Precedence constraints between the tasks
        for (int i = 0; i < nbTasks; ++i) {
            for (int s = 0; s < nbSuccessors[i]; ++s) {
                model.constraint(model.end(tasks[i]) <= model.start(tasks[successors[i][s]]));
            }
        }

        // Deadline constraint on the makespan
        makespan = model.max();
        for (int i = 0; i < nbTasks; ++i) {
            makespan.addOperand(model.end(tasks[i]));
        }
        model.constraint(makespan <= deadline);
        
        // Cumulative resource constraints
        for (int r = 0; r < nbResources; ++r) {
            HxExpression capacityRespected = model.createLambdaFunction([&](HxExpression t) {
                HxExpression totalWeight = model.sum();
                for (int i = 0; i < nbTasks; ++i) {
                    HxExpression taskWeight = weight[i][r] * model.contains(tasks[i], t);
                    totalWeight.addOperand(taskWeight);
                }
                return model.leq(totalWeight, capacity[r]);
            });
            model.constraint(model.and_(model.range(0, makespan), capacityRespected));
        }

        // Minimize the total cost
        totalCost = model.sum();
        for (int r = 0; r < nbResources; ++r) {
            totalCost.addOperand(resourceCost[r] * capacity[r]);
        }
        model.minimize(totalCost);

        model.close();

        // Parameterize the optimizer
        optimizer.getParam().setTimeLimit(timeLimit);

        optimizer.solve();
    }

    /* Write the solution in a file with the following format:
     *  - total cost
     *  - makespan
     *  - the capacity of each resource
     *  - for each task, the task id, the start and end times */
    void writeSolution(const std::string& fileName) {
        std::ofstream outfile(fileName.c_str());
        if (!outfile.is_open()) {
            std::cerr << "File " << fileName << " cannot be opened." << std::endl;
            exit(1);
        }
        std::cout << "Solution written in file " << fileName << std::endl;

        outfile << totalCost.getValue() << std::endl;
        outfile << makespan.getValue() << std::endl;
        for(int r = 0; r < nbResources; ++r) {
            outfile << capacity[r].getValue() << " ";
        }
        outfile << std::endl;
        for (int i = 0; i < nbTasks; ++i) {
            outfile << i + 1 << " " << tasks[i].getIntervalValue().getStart() << " "
                    << tasks[i].getIntervalValue().getEnd() << std::endl;
        }
        outfile.close();
    }
};

int main(int argc, char** argv) {
    if (argc < 2) {
        std::cout << "Usage: Racp instanceFile [outputFile] [timeLimit] [deadline]" << std::endl;
        exit(1);
    }

    const char* instanceFile = argv[1];
    const char* outputFile = argc > 2 ? argv[2] : NULL;
    const char* strTimeLimit = argc > 3 ? argv[3] : "60";
    const char* dline = argc > 4 ? argv[4] : NULL;

    Racp model;
    try {
        model.readInstance(instanceFile, dline);
        const int timeLimit = atoi(strTimeLimit);
        model.solve(timeLimit);
        if (outputFile != NULL)
            model.writeSolution(outputFile);
        return 0;
    } catch (const std::exception& e) {
        std::cerr << "An error occurred: " << e.what() << std::endl;
        return 1;
    }
}
