import hexaly.optimizer
import sys


# The input files follow the "Patterson" format
def read_instance(filename):
    with open(filename) as f:
        lines = f.readlines()

    first_line = lines[0].split()

    # Number of tasks
    nb_tasks = int(first_line[0])

    # Number of resources
    nb_resources = int(first_line[1])

    # Cost per unit of capacity of each resource
    resource_cost = [int(lines[1].split()[r]) for r in range(nb_resources)]

    # Duration of each task
    duration = [0 for _ in range(nb_tasks)]

    # Weight of resource r required for task i
    weight = [[] for _ in range(nb_tasks)]

    # Number of successors
    nb_successors = [0 for _ in range(nb_tasks)]

    # Successors of each task
    successors = [[] for _ in range(nb_tasks)]

    for i in range(nb_tasks):
        line = lines[i + 2].split()
        duration[i] = int(line[0])
        weight[i] = [int(line[r + 1]) for r in range(nb_resources)]
        nb_successors[i] = int(line[nb_resources + 1])
        successors[i] = [int(line[nb_resources + 2 + s]) - 1 for s in range(nb_successors[i])]

    # Trivial upper bound for the end times of the tasks
    horizon = sum(duration[i] for i in range(nb_tasks))

    return (nb_tasks, nb_resources, resource_cost, duration, weight, nb_successors, successors, horizon)


def main(instance_file, dline, output_file, time_limit):
    nb_tasks, nb_resources, resource_cost, duration, weight, nb_successors, successors, horizon = read_instance(
        instance_file)
    
    if dline is None:
        deadline = horizon
    else:
        deadline = int(dline)

    with hexaly.optimizer.HexalyOptimizer() as optimizer:
        #
        # Declare the optimization model
        #
        model = optimizer.model

        # Interval decision variables: time range of each task
        tasks = [model.interval(0, horizon) for _ in range(nb_tasks)]

        # Integer decision variables: capacity of each renewable resource
        capacity = [model.int(0, sum(weight[i][r] for i in range(nb_tasks))) for r in range(nb_resources)]

        # Task duration constraints
        for i in range(nb_tasks):
            model.constraint(model.length(tasks[i]) == duration[i])

        # Precedence constraints between the tasks
        for i in range(nb_tasks):
            for s in range(nb_successors[i]):
                model.constraint(model.end(tasks[i]) <= model.start(tasks[successors[i][s]]))

        # Deadline constraint on the makespan
        makespan = model.max([model.end(tasks[i]) for i in range(nb_tasks)])
        model.constraint(makespan <= deadline)

        # Cumulative resource constraints
        for r in range(nb_resources):
            capacity_respected = model.lambda_function(
                lambda t: model.sum(weight[i][r] * model.contains(tasks[i], t)
                                    for i in range(nb_tasks))
                <= capacity[r])
            model.constraint(model.and_(model.range(makespan), capacity_respected))

        # Minimize the total cost
        total_cost = model.sum(resource_cost[r] * capacity[r] for r in range(nb_resources))
        model.minimize(total_cost)
    
        model.close()
        # Parameterize the optimizer
        optimizer.param.time_limit = time_limit

        optimizer.solve()

        #
        # Write the solution in a file with the following format:
        # - total cost
        # - makespan
        # - the capacity of each resource
        # - for each task, the task id, the start and end times
        #
        if output_file != None:
            with open(output_file, "w") as f:
                print("Solution written in file", output_file)
                f.write(str(total_cost.value) + "\n")
                f.write(str(makespan.value) + "\n")
                for r in range(nb_resources):
                    f.write(str(capacity[r].value) + " ")
                f.write("\n")
                for i in range(nb_tasks):
                    f.write(str(i + 1) + " " + str(tasks[i].value.start()) + " " + str(tasks[i].value.end()))
                    f.write("\n")


if __name__ == '__main__':
    if len(sys.argv) < 2:
        print("Usage: python racp.py instance_file [output_file] [time_limit] [deadline]")
        sys.exit(1)
    instance_file = sys.argv[1]
    output_file = sys.argv[2] if len(sys.argv) >= 3 else None
    time_limit = int(sys.argv[3]) if len(sys.argv) >= 4 else 60
    dline = sys.argv[4] if len(sys.argv) >= 5 else None
    main(instance_file, dline, output_file, time_limit)
