using System;
using System.IO;
using System.Linq;
using Hexaly.Optimizer;

public class RcpspMultiMode : IDisposable
{
    // Number of tasks
    private int nbTasks;
    // Number of renewable resources
    private int nbRenewableResources;
    // Total number of resources (renewable and non-renewable)
    private int nbResources;
    // Number of available modes for each task
    private int[] nbModes;
    // Maximum capacity of each resource
    private int[] capacity;
    // Duration of each task and mode
    private int[][] duration;
    // Required resource weight for each task and mode
    private int[][][] weight;
    // Number of successors of each task
    private int[] nbSuccessors;
    // Successors of each task
    private int[][] successors;
    // Trivial upper bound for the start times of the tasks
    private int horizon = 0;

    // Hexaly Optimizer
    private HexalyOptimizer optimizer;
    // Decision variables: time range for each task and mode
    private HxExpression[][] tasksInMode;
    // Intermediate variables: boolean indicating for each pair (task, mode) whether it is active or not
    private HxExpression[][] presentMode;
    // Intermediate variables: time range of each task (hull of all RcpspMultiMode#tasksInMode)
    private HxExpression[] tasks;
    // Intermediate variables: number of active modes per task
    private HxExpression[] nbModesPerTask;
    // Objective = minimize the makespan: end of the last task of the last job
    private HxExpression makespan;

    public RcpspMultiMode(string fileName)
    {
        optimizer = new HexalyOptimizer();
    }

    public void Dispose()
    {
        optimizer.Dispose();
    }

    public void Solve(int timeLimit)
    {
        // Declare the optimization model
        HxModel model = optimizer.GetModel();

        // Optional interval decisions: time range for each task and mode
        tasksInMode = new HxExpression[nbTasks][];
        presentMode = new HxExpression[nbTasks][];

        // Hull
        tasks = new HxExpression[nbTasks];

        // Declare decisions
        for (int task = 0; task < nbTasks; task++)
        {
            tasksInMode[task] = new HxExpression[nbModes[task]];
            presentMode[task] = new HxExpression[nbModes[task]];
            for (int mode = 0; mode < nbModes[task]; mode++)
            {
                tasksInMode[task][mode] = model.OptionalInterval(0, horizon);
                presentMode[task][mode] = model.Presence(tasksInMode[task][mode]);
            }
            tasks[task] = model.Hull(tasksInMode[task]);
        }

        // Build makespan
        makespan = model.Max();
        for (int task = 0; task < nbTasks; task++)
        {
            makespan.AddOperand(model.End(tasks[task]));
        }

        // Declare constraints on tasks
        nbModesPerTask = new HxExpression[nbTasks];
        for (int task = 0; task < nbTasks; task++)
        {
            // Constraints: Task duration
            for (int mode = 0; mode < nbModes[task]; mode++)
            {
                model.Constraint(model.If(
                        presentMode[task][mode], 
                        model.Eq(model.Length(tasksInMode[task][mode]), duration[task][mode]), 
                        1
                ));
            }

            // Constraints: Precedence between the tasks
            for (int successor = 0; successor < nbSuccessors[task]; successor++)
            {
                model.Constraint(model.Lt(tasks[task], tasks[successors[task][successor]]));
            }

            // Constraints: Exactly one active mode for each task
            nbModesPerTask[task] = model.Sum();
            for (int mode = 0; mode < nbModes[task]; mode++)
            {
                nbModesPerTask[task].AddOperand(presentMode[task][mode]);
            }
            model.Constraint(model.Eq(nbModesPerTask[task], 1));
        }

        // Constraints: Renewable resources
        for (int resource = 0; resource < nbRenewableResources; resource++) 
        {
            HxExpression capacityRespected = model.LambdaFunction(time =>
            {
                HxExpression totalWeight = model.Sum();
                for (int task = 0; task < nbTasks; task++)
                {
                    for (int mode = 0; mode < nbModes[task]; mode++)
                    {
                        totalWeight.AddOperand(model.Prod(
                                weight[task][mode][resource],
                                model.Contains(tasksInMode[task][mode], time)));
                    }
                }
                return model.Leq(totalWeight, capacity[resource]);
            });
            model.Constraint(model.And(model.Range(0, makespan), capacityRespected));
        }

        // Constraints: Non-renewable resources
        for (int resource = nbRenewableResources; resource < nbResources; resource++) 
        {
            HxExpression totalWeight = model.Sum();
                for (int task = 0; task < nbTasks; task++)
                {
                    for (int mode = 0; mode < nbModes[task]; mode++)
                    {
                        totalWeight.AddOperand(model.Prod(
                                weight[task][mode][resource],
                                presentMode[task][mode]));
                    }
                }
            model.Constraint(model.Leq(totalWeight, capacity[resource]));
        }

        // Objective: Minimize makespan
        model.Minimize(makespan);

        model.Close();

        // Parameterize the optimizer
        optimizer.GetParam().SetTimeLimit(timeLimit);

        optimizer.Solve();
    }

    /*
     * Write the solution in a file with the following format:
     *  - total makespan
     *  - for each task, the task ID, the active mode ID, the start and end times
     */
    public void WriteSolution(string fileName)
    {
        using (StreamWriter output = new StreamWriter(fileName))
        {
            Console.WriteLine("Solution written in file " + fileName);
            output.WriteLine(makespan.GetValue());
            
            int activeModeId = -1;
            for (int task = 0; task < nbTasks; task++)
            {
                for (int mode = 0; mode < nbModes[task]; mode++)
                {
                    if (presentMode[task][mode].GetValue() == 1)
                    {
                        activeModeId = mode;
                        break;
                    }
                }
                output.Write((task + 1) + " " + (activeModeId + 1) + " "
                        + tasks[task].GetIntervalValue().Start() + " "
                        + tasks[task].GetIntervalValue().End());
                output.WriteLine();
            }
        }
    }

    string[] SplitInput(StreamReader input)
    {
        string line = input.ReadLine();
        if (line == null)
            return new string[0];
        return line.Split(new[] { ' ' }, StringSplitOptions.RemoveEmptyEntries);
    }

    private void ReadInstance(string fileName)
    {
        using (StreamReader input = new StreamReader(fileName))
        {
            string[] splitted;

            // Parse number of tasks
            for (int i = 0; i < 5; ++i)
            {
                input.ReadLine();
            }
            splitted = input.ReadLine().Split(':');

            nbTasks = int.Parse(splitted[1]);

            // Parse number of resources
            for (int i = 0; i < 2; ++i)
            {
                input.ReadLine();
            }
            splitted = input.ReadLine().Split(':');
            nbRenewableResources = int.Parse(new string(splitted[1]
                        .SkipWhile(c => !char.IsDigit(c))
                        .TakeWhile(c => char.IsDigit(c))
                        .ToArray()));

            splitted = input.ReadLine().Split(':');
            int nbNonRenewableResources = int.Parse(new string(splitted[1]
                        .SkipWhile(c => !char.IsDigit(c))
                        .TakeWhile(c => char.IsDigit(c))
                        .ToArray()));

            nbResources = nbRenewableResources + nbNonRenewableResources;

            // Parse successors of each task
            for (int i = 0; i < 8; ++i)
            {
                input.ReadLine();
            }
            splitted = SplitInput(input);

            nbModes = new int[nbTasks];
            nbSuccessors = new int[nbTasks];
            successors = new int[nbTasks][];
            
            int task = int.Parse(splitted[0]) - 1;
            while (true)
            {
                nbModes[task] = int.Parse(splitted[1]);
                nbSuccessors[task] = int.Parse(splitted[2]);
                successors[task] = new int[nbSuccessors[task]];
                for (int successor = 0; successor < nbSuccessors[task]; successor++)
                {
                    successors[task][successor] = int.Parse(splitted[3 + successor]) - 1;
                }
                if (task + 1 == nbTasks)
                {
                     break;
                }
                splitted = SplitInput(input);
                task = int.Parse(splitted[0]) - 1;
            }

            // Parse tasks durations per mode AND consumed resource weight per mode for each task
            for (int i = 0; i < 4; ++i)
            {
                input.ReadLine();
            }
            splitted = SplitInput(input);

            task = int.Parse(splitted[0]) - 1;
            duration = new int[nbTasks][];
            weight = new int[nbTasks][][];
            int baseIndex;
            int modeId;
            int maxDuration;

            while (true)
            {
                weight[task] = new int[nbModes[task]][];
                duration[task] = new int[nbModes[task]];
                maxDuration = 0;
                for (int mode = 0; mode < nbModes[task]; mode++)
                {
                    baseIndex = mode == 0 ? 1 : 0;
                    modeId = int.Parse(splitted[baseIndex]) - 1;
                    duration[task][modeId] = int.Parse(splitted[baseIndex + 1]);
                    weight[task][modeId] = new int[nbResources];
                    maxDuration = Math.Max(maxDuration, duration[task][modeId]);
                    for (int resourceId = 0; resourceId < nbResources; resourceId++)
                    {
                        weight[task][modeId][resourceId] = int.Parse(splitted[baseIndex + 2 + resourceId]);
                    }
                    splitted = SplitInput(input);
                }
                horizon += maxDuration;
                if (task + 1 == nbTasks)
                {
                    break;
                }
                task = int.Parse(splitted[0]) - 1;
            }

            // Parse maximum capacity for each resource
            for (int i = 0; i < 2; ++i)
            {
                input.ReadLine();
            }

            capacity = new int[nbResources];
            splitted = SplitInput(input);

            for (int resource = 0; resource < nbResources; resource++)
            {
                capacity[resource] = int.Parse(splitted[resource]);
            }
        }
    }

    public static void Main(string[] args)
    {
        if (args.Length < 1)
        {
            Console.WriteLine("Usage: RcpspMultiMode instanceFile [outputFile] [timeLimit]");
            System.Environment.Exit(1);
        }

        string instanceFile = args[0];
        string outputFile = args.Length > 1 ? args[1] : null;
        string strTimeLimit = args.Length > 2 ? args[2] : "60";

        using (RcpspMultiMode model = new RcpspMultiMode(instanceFile))
        {
            model.ReadInstance(instanceFile);
            model.Solve(int.Parse(strTimeLimit));
            if (outputFile != null)
                model.WriteSolution(outputFile);
        }
    }
}
