import java.util.*;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.io.*;
import com.hexaly.optimizer.*;

public class RcpspMultiMode {
    // Number of tasks
    private int nbTasks;
    // Number of renewable resources
    private int nbRenewableResources;
    // Total number of resources (renewable and non-renewable)
    private int nbResources;
    // Number of available modes for each task
    private int[] nbModes;
    // Maximum capacity of each resource
    private int[] capacity;
    // Duration of each task and mode
    private int[][] duration;
    // Required resource weight for each task and mode
    private int[][][] weight;
    // Number of successors of each task
    private int[] nbSuccessors;
    // Successors of each task
    private int[][] successors;
    // Trivial upper bound for the start times of the tasks
    private int horizon = 0;

    // Hexaly Optimizer
    final HexalyOptimizer optimizer;
    // Decision variables: time range for each task and mode
    private HxExpression[][] tasksInMode;
    // Intermediate variables: boolean indicating for each pair (task, mode) whether it is active or not
    private HxExpression[][] presentMode;
    // Intermediate variables: time range of each task (hull of all RcpspMultiMode#tasksInMode)
    private HxExpression[] tasks;
    // Intermediate variables: number of active modes per task
    private HxExpression[] nbModesPerTask;
    // Objective = minimize the makespan: end of the last task of the last job
    private HxExpression makespan;

    public RcpspMultiMode(HexalyOptimizer optimizer, String fileName) throws IOException {
        this.optimizer = optimizer;
    }

    public void solve(int timeLimit) {
        // Declare the optimization model
        HxModel model = optimizer.getModel();

        // Optional interval decisions: time range for each task and mode
        tasksInMode = new HxExpression[nbTasks][];
        presentMode = new HxExpression[nbTasks][];

        // Hull
        tasks = new HxExpression[nbTasks];

        // Declare decisions
        for (int task = 0; task < nbTasks; task++) {
            tasksInMode[task] = new HxExpression[nbModes[task]];
            presentMode[task] = new HxExpression[nbModes[task]];
            for (int mode = 0; mode < nbModes[task]; mode++) {
                tasksInMode[task][mode] = model.optionalIntervalVar(0, horizon);
                presentMode[task][mode] = model.presence(tasksInMode[task][mode]);
            }
            tasks[task] = model.hull(tasksInMode[task]);
        }

        // Build makespan
        makespan = model.max();
        for (int task = 0; task < nbTasks; task++) {
            makespan.addOperand(model.end(tasks[task]));
        }

        // Declare constraints on tasks
        nbModesPerTask = new HxExpression[nbTasks];
        for (int task = 0; task < nbTasks; task++) {
            // Constraints: Task duration
            for (int mode = 0; mode < nbModes[task]; mode++) {
                model.constraint(model.iif(
                        presentMode[task][mode], 
                        model.eq(model.length(tasksInMode[task][mode]), duration[task][mode]), 
                        1
                ));
            }

            // Constraints: Precedence between tasks
            for (int successor = 0; successor < nbSuccessors[task]; successor++) {
                model.constraint(model.lt(tasks[task], tasks[successors[task][successor]]));
            }

            // Constraints: Exactly one active mode for each task
            nbModesPerTask[task] = model.sum();
            for (int mode = 0; mode < nbModes[task]; mode++) {
                nbModesPerTask[task].addOperand(presentMode[task][mode]);
            }
            model.constraint(model.eq(nbModesPerTask[task], 1));
        }

        // Constraints: Renewable resources
        for (int r = 0; r < nbRenewableResources; r++) {
            final int resource = r;
            HxExpression capacityRespected = model.lambdaFunction(time -> {
                HxExpression totalWeight = model.sum();
                for (int task = 0; task < nbTasks; task++) {
                    for (int mode = 0; mode < nbModes[task]; mode++) {
                        totalWeight.addOperand(model.prod(
                                weight[task][mode][resource],
                                model.contains(tasksInMode[task][mode], time)));
                    }
                }
                return model.leq(totalWeight, capacity[resource]);
            });
            model.constraint(model.and(model.range(0, makespan), capacityRespected));
        }

        // Constraints: Non-renewable resources
        for (int resource = nbRenewableResources; resource < nbResources; resource++) {
            HxExpression totalWeight = model.sum();
            for (int task = 0; task < nbTasks; task++) {
                for (int mode = 0; mode < nbModes[task]; mode++) {
                    totalWeight.addOperand(model.prod(weight[task][mode][resource], presentMode[task][mode]));
                }
            }
            model.constraint(model.leq(totalWeight, capacity[resource]));
        }

        // Objective: Minimize makespan
        model.minimize(makespan);

        model.close();

        // Parameterize the optimizer
        optimizer.getParam().setTimeLimit(timeLimit);

        optimizer.solve();
    }

    /*
     * Write the solution in a file with the following format:
     * - total makespan
     * - for each task, the task ID, the active mode ID, the start and end times
     */
    public void writeSolution(String fileName) throws IOException {
        try (PrintWriter output = new PrintWriter(fileName)) {
            System.out.println("Solution written in file " + fileName);

            output.println(makespan.getValue());

            int activeModeId = -1;
            for (int task = 0; task < nbTasks; ++task) {
                for (int mode = 0; mode < nbModes[task]; mode++) {
                    if (presentMode[task][mode].getValue() == 1) {
                        activeModeId = mode;
                        break;
                    }
                }
                output.println((task + 1) + " " + (activeModeId + 1) + " "
                        + tasks[task].getIntervalValue().getStart() + " "
                        + tasks[task].getIntervalValue().getEnd());
            }
        }
    }

    private void readInstance(String fileName) throws IOException {
        try (Scanner input = new Scanner(new File(fileName))) {
            input.useLocale(Locale.ROOT);

            // Parse number of tasks
            for (int i = 0; i < 5; i++) {
                input.nextLine();
            }
            String str = input.nextLine();

            String[] splitted = str.split(":");
            nbTasks = Integer.valueOf(splitted[1].trim());

            // Parse number of resources
            for (int i = 0; i < 2; i++) {
                input.nextLine();
            }

            str = input.nextLine();
            splitted = str.split(":");
            Matcher matcher = Pattern.compile("\\d+").matcher(splitted[1].trim());
            matcher.find();
            nbRenewableResources = Integer.valueOf(matcher.group());

            str = input.nextLine();
            splitted = str.split(":");
            matcher = Pattern.compile("\\d+").matcher(splitted[1].trim());
            matcher.find();
            int nbNonRenewableResources = Integer.valueOf(matcher.group());
            nbResources = nbRenewableResources + nbNonRenewableResources;

            // Parse successors of each task
            for (int i = 0; i < 7; i++) {
                input.nextLine();
            }
            str = input.nextLine();

            nbModes = new int[nbTasks];
            nbSuccessors = new int[nbTasks];
            successors = new int[nbTasks][];
            int task = input.nextInt() - 1;
            while (true) {
                nbModes[task] = input.nextInt();
                nbSuccessors[task] = input.nextInt();
                successors[task] = new int[nbSuccessors[task]];
                for (int successor = 0; successor < nbSuccessors[task]; successor++) {
                    successors[task][successor] = input.nextInt() - 1;
                }
                if (task + 1 == nbTasks) {
                     break;
                }
                task = input.nextInt() - 1;
            }

            // Parse tasks durations per mode AND consumed resource weight per mode for each task
            for (int i = 0; i < 4; i++) {
                input.nextLine();
            }
            str = input.nextLine();

            task = input.nextInt() - 1;
            duration = new int[nbTasks][];
            weight = new int[nbTasks][][];
            int modeId;
            int maxDuration;

            while (true) {
                weight[task] = new int[nbModes[task]][];
                duration[task] = new int[nbModes[task]];
                maxDuration = 0;
                for (int mode = 0; mode < nbModes[task]; mode++) {
                    modeId = input.nextInt() - 1;
                    duration[task][modeId] = input.nextInt();
                    weight[task][modeId] = new int[nbResources];
                    maxDuration = Math.max(maxDuration, duration[task][modeId]);
                    for (int resourceId = 0; resourceId < nbResources; resourceId++) {
                        weight[task][modeId][resourceId] = input.nextInt();
                    }
                }
                horizon += maxDuration;
                if (task + 1 == nbTasks) {
                    break;
                }
                task = input.nextInt() - 1;
            }

            // Parse maximum capacity for each resource
            for (int i = 0; i < 3; i++) {
                input.nextLine();
            }
            str = input.nextLine();
            capacity = new int[nbResources];
            for (int resource = 0; resource < nbResources; resource++) {
                capacity[resource] = input.nextInt();
            }
        }
    }

    public static void main(String[] args) {
        if (args.length < 1) {
            System.out.println("Usage: java RcpspMultiMode instanceFile [outputFile] [timeLimit]");
            System.exit(1);
        }

        String instanceFile = args[0];
        String outputFile = args.length > 1 ? args[1] : null;
        String strTimeLimit = args.length > 2 ? args[2] : "60";

        try (HexalyOptimizer optimizer = new HexalyOptimizer()) {
            RcpspMultiMode model = new RcpspMultiMode(optimizer, instanceFile);
            model.readInstance(instanceFile);
            model.solve(Integer.parseInt(strTimeLimit));
            if (outputFile != null) {
                model.writeSolution(outputFile);
            }
        } catch (Exception ex) {
            System.err.println(ex);
            ex.printStackTrace();
            System.exit(1);
        }
    }
}
